package net.doo.snap.lib.billing;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import net.doo.snap.lib.util.billing.IabHelper;

/**
 * Manages user's purchases and access to in-app billing service
 */
public interface BillingManager {

    /**
     * Opens connection to billing service and asynchronously updates the status of user's purchases.
     * When you're done with {@link net.doo.snap.lib.billing.BillingManager}, call {@link #dispose()}.
     * {@link net.doo.snap.lib.util.billing.IabHelper.OnIabSetupFinishedListener} callbacks will be executed in main thread
     *
     * @param context
     * @param listener
     */
    public void initialize(Context context, IabHelper.OnIabSetupFinishedListener listener);

    /**
     * Checks locally if item is available for user. Can be used from main thread.
     *
     * @param itemId id of SKU being checked
     * @return {@code true} if item is available. {@code false} otherwise.
     */
    public boolean isItemAvailable(String itemId);

    /**
     * @param itemId SKU id
     *
     * @return item price
     */
    public String getItemPrice(String itemId);

    /**
     * Updates local state of purchased items. This is a blocking request, do not use it in main thread.
     */
    public void refreshItems();

    /**
     * Updates local state of purchased items. This is not a blocking request, do use it in main thread.
     * {@link net.doo.snap.lib.billing.BillingManagerImpl.RefreshItemsListener} callbacks will be executed in main thread
     *
     * @param listener
     */
    public void refreshItemsAsync(BillingManagerImpl.RefreshItemsListener listener);

    /**
     * Starts Google Play billing workflow for provided SKU id.
     * {@link net.doo.snap.lib.util.billing.IabHelper.OnIabPurchaseFinishedListener} callbacks will be executed in main thread
     *
     * @param activity
     * @param itemId SKU id
     * @param listener
     */
    public void startBillingWorkflowForItem(Activity activity, String itemId, IabHelper.OnIabPurchaseFinishedListener listener);

    /**
     * Handles result posted to {@link android.app.Activity#onActivityResult(int, int, android.content.Intent)}.
     * Either consumes result by returning {@code true}, or ignores it by returning {@code false}.
     *
     * @param requestCode
     * @param data
     * @param resultCode
     *
     * @return {@code true} if result was processed by {@link net.doo.snap.lib.billing.BillingManager}.
     * {@code false} if result was ignored by {@link net.doo.snap.lib.billing.BillingManager}
     */
    public boolean handleResult(int requestCode, int resultCode, Intent data);

    /**
     * Releases connections and resources held by {@link net.doo.snap.lib.billing.BillingManager}
     */
    public void dispose();

}
