package net.doo.snap.lib.persistence;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;

import com.google.inject.Inject;

import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.bitmap.BitmapUtils;
import net.doo.snap.lib.util.log.DebugLog;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Creates {@link net.doo.snap.lib.persistence.Page}s and saves their content in file system
 *
 * @see PageStoreStrategy
 */
public class PageFactory {

    private static final ExecutorService executor = Executors.newSingleThreadExecutor();

    @Inject
    private BitmapLruCache bitmapLruCache;

    private PageStoreStrategy pageStoreStrategy;

    @Inject
    public PageFactory(PageStoreStrategy pageStoreStrategy) {
        this.pageStoreStrategy = pageStoreStrategy;
    }

    /**
     * Builds new {@link net.doo.snap.lib.persistence.Page} from provided data. Result represented as
     * {@link net.doo.snap.lib.persistence.PageFactory.Result} class.
     *
     * @param image        original image bytes
     * @param screenWidth  width of the device screen
     * @param screenHeight height of the device screen
     * @return {@link net.doo.snap.lib.persistence.PageFactory.Result} which contains information about
     * the created {@link net.doo.snap.lib.persistence.Page}
     * @throws IOException in case if files associated with {@link net.doo.snap.lib.persistence.Page} could
     *                     not be created.
     */
    public Result buildPage(byte[] image, int screenWidth, int screenHeight) throws IOException {
        Page page = new Page();
        File pageDir = pageStoreStrategy.getPageDir(page.getId());
        FileUtils.forceMkdir(pageDir);

        File filteredDir = FileUtils.getFile(
                pageDir,
                Page.FILTERED_FOLDER
        );
        FileUtils.forceMkdir(filteredDir);

        FileUtils.writeByteArrayToFile(new File(pageDir, Page.ImageType.ORGINAL.getFileName()), image);

        final Bitmap preview = BitmapUtils.createPreview(image, screenWidth / 2, screenHeight / 2);

        page.setImageSize(Page.ImageType.OPTIMIZED_PREVIEW, preview.getWidth(), preview.getHeight());

        final String path = page.getFilePath(pageStoreStrategy, Page.ImageType.PREVIEW);
        if (path == null) {
            throw new IOException("can't create page preview");
        }

        bitmapLruCache.put(path, preview);

        executor.execute(new Runnable() {
            @Override
            public void run() {
                FileOutputStream out = null;
                try {
                    out = new FileOutputStream(path);
                    preview.compress(Bitmap.CompressFormat.JPEG, 90, out);
                } catch (IOException e) {
                    DebugLog.logException(e);
                } finally {
                    IOUtils.closeQuietly(out);
                }
            }
        });

        return new Result(page, preview);
    }

    /**
     * Result of {@link net.doo.snap.lib.persistence.Page} creation.
     */
    public static class Result {

        /**
         * Newly created {@link net.doo.snap.lib.persistence.Page} with unique id
         */
        public final Page page;

        /**
         * Preview-size {@link android.graphics.Bitmap} for {@link net.doo.snap.lib.persistence.Page}
         */
        public final Bitmap preview;

        public Result(Page page, Bitmap preview) {
            this.page = page;
            this.preview = preview;
        }
    }
}
