package net.doo.snap.lib.util.ui;

import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;

/**
 * Composition of {@link android.view.View.OnTouchListener}s. Propagates touch event to listeners
 * in order of their registration. As soon as one of the listeners consumes the event, it won't be
 * propagated any further until next {@link android.view.MotionEvent#ACTION_DOWN} event.
 */
public class CompositeTouchListener implements View.OnTouchListener {

    private ArrayList<View.OnTouchListener> listeners = new ArrayList<View.OnTouchListener>();

    /**
     * Registers new {@link android.view.View.OnTouchListener}
     *
     * @param listener {@link android.view.View.OnTouchListener} which will receive callbacks
     * @return same {@link net.doo.snap.lib.util.ui.CompositeTouchListener}
     */
    public CompositeTouchListener addListener(@NotNull View.OnTouchListener listener) {
        listeners.add(listener);

        return this;
    }

    /**
     * Registers new {@link android.view.GestureDetector}
     *
     * @param gestureDetector {@link android.view.GestureDetector} which will receive callbacks
     * @return same {@link net.doo.snap.lib.util.ui.CompositeTouchListener}
     */
    public CompositeTouchListener addGestureDetector(@NotNull GestureDetector gestureDetector) {
        listeners.add(new GestureDetectorListener(gestureDetector));

        return this;
    }

    /**
     * Registers new {@link android.view.ScaleGestureDetector}
     *
     * @param scaleGestureDetector {@link android.view.ScaleGestureDetector} which will receive callbacks
     * @return same {@link net.doo.snap.lib.util.ui.CompositeTouchListener}
     */
    public CompositeTouchListener addScaleGestureDetector(@NotNull ScaleGestureDetector scaleGestureDetector) {
        listeners.add(new ScaleGestureDetectorListener(scaleGestureDetector));

        return this;
    }

    @Override
    public boolean onTouch(View view, MotionEvent event) {
        boolean consumed = false;
        for (View.OnTouchListener listener : listeners) {
            consumed |= listener.onTouch(view, event);
        }

        return consumed;
    }

    /**
     * Util class that wraps {@link android.view.GestureDetector} into {@link android.view.View.OnTouchListener}
     */
    static class GestureDetectorListener implements View.OnTouchListener {

        private final GestureDetector gestureDetector;

        public GestureDetectorListener(@NotNull GestureDetector gestureDetector) {
            this.gestureDetector = gestureDetector;
        }

        @Override
        public boolean onTouch(View view, MotionEvent event) {
            return gestureDetector.onTouchEvent(event);
        }
    }

    /**
     * Util class that wraps {@link android.view.ScaleGestureDetector} into {@link android.view.View.OnTouchListener}
     */
    static class ScaleGestureDetectorListener implements View.OnTouchListener {

        private final ScaleGestureDetector scaleGestureDetector;

        public ScaleGestureDetectorListener(ScaleGestureDetector scaleGestureDetector) {
            this.scaleGestureDetector = scaleGestureDetector;
        }

        @Override
        public boolean onTouch(View view, MotionEvent event) {
            return scaleGestureDetector.onTouchEvent(event);
        }

    }

}
