package net.doo.snap.lib.persistence;

import android.graphics.Point;
import android.graphics.PointF;
import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.SparseArray;

import org.apache.commons.io.FileUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

/**
 * Model for representing snapped instance and associated meta info.
 *
 * @see DocumentDraft
 * @see net.doo.snap.lib.persistence.PageFactory
 * @see net.doo.snap.lib.persistence.Signature
 * @see net.doo.snap.lib.persistence.Page.ImageType
 */
public final class Page implements Parcelable {

    public static final String FILTERED_FOLDER = "filtered";

    private final String id;
    private final Bundle parameters;
    private OptimizationType optimizationType = OptimizationType.NONE;
    private RotationType rotationType = RotationType.ROTATION_0;
    private List<PointF> polygon = new ArrayList<PointF>() {
        {
            add(new PointF(0, 0));
            add(new PointF(1, 0));
            add(new PointF(1, 1));
            add(new PointF(0, 1));
        }
    };
    private SparseArray<Point> imageSizes = new SparseArray<Point>();
    private List<Signature> signatures = new ArrayList<Signature>();
    private List<Annotation> annotations = new ArrayList<Annotation>();

    private boolean processed = false;

    Page() {
        this.id = UUID.randomUUID().toString();
        parameters = new Bundle();
    }

    public Page(String id) {
        this.id = id;
        parameters = new Bundle();
    }

    public String getId() {
        return id;
    }

    public OptimizationType getOptimizationType() {
        return optimizationType;
    }

    public void setOptimizationType(OptimizationType optimizationType) {
        this.optimizationType = optimizationType;
    }

    public RotationType getRotationType() {
        return rotationType;
    }

    public void setRotationType(RotationType rotationType) {
        this.rotationType = rotationType;
    }

    /**
     * @return polygon which crops page preview from original image
     */
    public List<PointF> getPolygon() {
        if (polygon == null) {
            return Collections.emptyList();
        }

        return polygon;
    }

    /**
     * Sets polygon which crops page preview from original image. {@code null} and
     * polygons with the number of points different from 4 are also ignored
     */
    public void setPolygon(List<PointF> polygon) {
        if (polygon == null || polygon.size() != 4) {
            return;
        }
        this.polygon = polygon;
    }

    /**
     * @param pageStoreStrategy
     * @param type
     * @return path including page folder and file name depending on image type
     * @throws java.io.IOException if {@link java.io.File} for {@link net.doo.snap.lib.persistence.Page} can't be opened
     */
    public String getFilePath(PageStoreStrategy pageStoreStrategy, ImageType type) throws IOException {
        return getFile(pageStoreStrategy, type).getPath();
    }

    /**
     * @param pageStoreStrategy
     * @param pageId
     * @param type
     * @return path including page folder with specified id and file name depending on image type or {@code null}
     * if no file for specified pageId was found
     * @throws java.io.IOException if {@link java.io.File} for {@link net.doo.snap.lib.persistence.Page} can't be opened
     */
    public static String getFilePath(PageStoreStrategy pageStoreStrategy, String pageId, ImageType type) throws IOException {
        File file = FileUtils.getFile(pageStoreStrategy.getPagesDir(), pageId, type.getFileName());
        return file.getPath();
    }

    /**
     * @param pageStoreStrategy
     * @param type
     * @return image {@link java.io.File} of provided {@link net.doo.snap.lib.persistence.Page.ImageType} for this {@link net.doo.snap.lib.persistence.Page}
     * @throws java.io.IOException if {@link java.io.File} for {@link net.doo.snap.lib.persistence.Page} can't be opened
     */
    public File getFile(PageStoreStrategy pageStoreStrategy, ImageType type) throws IOException {
        return FileUtils.getFile(pageStoreStrategy.getPagesDir(), id, type.getFileName());
    }

    /**
     * @param optimizationType {@link net.doo.snap.lib.persistence.OptimizationType} used for filtration of image
     * @return file name of filtered image
     * @throws java.io.IOException if {@link java.io.File} for {@link net.doo.snap.lib.persistence.Page} can't be opened
     */
    public File getFilteredFile(PageStoreStrategy pageStoreStrategy, OptimizationType optimizationType) throws IOException {
        return FileUtils.getFile(
                pageStoreStrategy.getPageDir(id),
                FILTERED_FOLDER,
                optimizationType.getName()
        );
    }

    /**
     * Saves information about image size for particular {@link net.doo.snap.lib.persistence.Page.ImageType}
     *
     * @param imageType
     * @param width
     * @param height
     */
    void setImageSize(ImageType imageType, int width, int height) {
        imageSizes.put(imageType.ordinal(), new Point(width, height));
    }

    /**
     * @param imageType
     * @return {@link android.graphics.Point} which represents size of image for particular {@link net.doo.snap.lib.persistence.Page.ImageType}
     */
    public Point getImageSize(ImageType imageType) {
        return imageSizes.get(imageType.ordinal());
    }

    /**
     * @return {@code true} if {@link Page} original image was processed. {@code false} otherwise.
     */
    public boolean isProcessed() {
        return processed;
    }

    /**
     * @param processed {@code true} if {@link Page} original image was processed. {@code false} otherwise.
     */
    public void setProcessed(boolean processed) {
        this.processed = processed;
    }

    /**
     * @return additional page parameters which could be stored as key-value set
     */
    public Bundle getParameters() {
        return parameters;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Page)) return false;
        Page page = (Page) o;

        return id.equals(page.id);
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

    /**
     * Types of images to be saved on disk
     */
    public enum ImageType {
        ORGINAL("original"),
        PREVIEW("preview"),
        OPTIMIZED_PREVIEW("optimized_preview"),
        OPTIMIZED("optimized"),
        OVERLAY("overlay"),
        OVERLAY_TMP("overlay_tmp"),
        COMBINED("combined");

        private String fileName;

        ImageType(String fileName) {
            this.fileName = fileName;
        }

        /**
         * @return file name to be saved on disk for this image type
         */
        public String getFileName() {
            return fileName;
        }

    }

    /**
     * Adds {@link net.doo.snap.lib.persistence.Signature} of the page. Overwrites signature with the same Id
     */
    public void addSignature(@NotNull Signature newSignature) {
        final int index = signatures.indexOf(newSignature);
        if (index < 0) {
            signatures.add(newSignature);
        } else {
            signatures.set(index, newSignature);
        }
    }

    /**
     * @return {@link net.doo.snap.lib.persistence.Signature} if there is one associated with the current page by position
     */
    @NotNull
    public Signature getSignature(int position) {
        return signatures.get(position);
    }

    /**
     * @return {@link net.doo.snap.lib.persistence.Signature} if there is one associated with the current page by given id, {@code null} otherwise
     */
    public Signature getSignature(String signatureId) {
        for (Signature signature : signatures) {
            if (signature.getId().equals(signatureId)) {
                return signature;
            }
        }

        return null;
    }

    /**
     * Detaches {@link net.doo.snap.lib.persistence.Signature} from page
     */
    public void removeSignature(@NotNull Signature signature) {
        signatures.remove(signature);
    }

    /**
     * Detaches {@link net.doo.snap.lib.persistence.Signature} with given id from page.
     */
    public void removeSignature(@NotNull String signatureId) {
        final Iterator<Signature> iterator = signatures.iterator();
        while (iterator.hasNext()) {
            Signature signature = iterator.next();

            if (signatureId.equals(signature.getId())) {
                iterator.remove();
            }
        }
    }

    /**
     * @return count of signatures
     */
    public int getSignaturesCount() {
        return signatures.size();
    }

    /**
     * Adds {@link net.doo.snap.lib.persistence.Annotation} to the page. Overwrites annotation with the same id.
     */
    public void addAnnotation(Annotation annotation) {
        final int index = annotations.indexOf(annotation);
        if (index < 0) {
            annotations.add(annotation);
        } else {
            annotations.set(index, annotation);
        }
    }

    /**
     * @return {@link net.doo.snap.lib.persistence.Annotation} at given position
     */
    public Annotation getAnnotation(int position) {
        return annotations.get(position);
    }

    /**
     * @return {@link net.doo.snap.lib.persistence.Annotation} with given id or {@code null} if it can't be found
     */
    public Annotation getAnnotation(String annotationId) {
        for (Annotation annotation : annotations) {
            if (annotation.getId().equals(annotationId)) {
                return annotation;
            }
        }

        return null;
    }

    /**
     * Detaches {@link net.doo.snap.lib.persistence.Annotation} from page
     */
    public void removeAnnotation(Annotation annotation) {
        annotations.remove(annotation);
    }

    /**
     * @return number of {@link net.doo.snap.lib.persistence.Annotation} attached to page
     */
    public int getAnnotationsCount() {
        return annotations.size();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.id);
        dest.writeBundle(parameters);
        dest.writeInt(this.optimizationType == null ? -1 : this.optimizationType.ordinal());
        dest.writeInt(this.rotationType == null ? -1 : this.rotationType.ordinal());
        dest.writeTypedList(polygon);
        dest.writeSparseArray((SparseArray) imageSizes);
        dest.writeTypedList(signatures);
        dest.writeTypedList(annotations);
    }

    private Page(Parcel in) {
        this.id = in.readString();
        parameters = in.readBundle();
        int tmpOptimizationType = in.readInt();
        this.optimizationType = tmpOptimizationType == -1 ? null : OptimizationType.values()[tmpOptimizationType];
        int tmpRotationType = in.readInt();
        this.rotationType = tmpRotationType == -1 ? null : RotationType.values()[tmpRotationType];
        in.readTypedList(polygon, PointF.CREATOR);
        this.imageSizes = in.readSparseArray(SparseArray.class.getClassLoader());
        in.readTypedList(signatures, Signature.CREATOR);
        in.readTypedList(annotations, Annotation.CREATOR);
    }

    public static Parcelable.Creator<Page> CREATOR = new Parcelable.Creator<Page>() {
        public Page createFromParcel(Parcel source) {
            return new Page(source);
        }

        public Page[] newArray(int size) {
            return new Page[size];
        }
    };
}

