package net.doo.snap.lib.ui.util;

import android.animation.Animator;
import android.graphics.Bitmap;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;
import android.view.View;
import android.view.ViewTreeObserver;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Contains tools for usage in {@link android.view.View} animations and transformations
 */
public class ViewUtils {

    public static final float ALPHA_DEFAULT = 1f;
    public static final float ALPHA_TRANSPARENT = 0f;

    public static final float SCALE_DEFAULT = 1f;
    public static final float TRANSLATION_DEFAULT = 0f;

    private static final int TRANSLATION_X = 0;
    private static final int TRANSLATION_Y = 1;
    private static final int SCALE_X = 2;
    private static final int SCALE_Y = 3;
    private static final int ROTATION = 4;
    private static final int ALPHA = 5;

    /**
     * Saves {@link android.view.View} transformation parameters (translation, scale, rotation, alpha), so
     * they can later be restored with {@link #restoreViewTransformation(android.view.View)}. Note, that this
     * implementation uses {@link View#setTag(Object)} method to save parameters, so any existing tags will be overwritten.
     */
    public static void saveViewTransformation(View view) {
        float[] values = new float[6];
        values[TRANSLATION_X] = view.getTranslationX();
        values[TRANSLATION_Y] = view.getTranslationY();
        values[SCALE_X] = view.getScaleX();
        values[SCALE_Y] = view.getScaleY();
        values[ROTATION] = view.getRotation();
        values[ALPHA] = view.getAlpha();

        view.setTag(values);
    }

    /**
     * Restores transformation values.
     *
     * @throws java.lang.IllegalStateException if values were corrupted (by direct invocation of {@link View#setTag(Object)})
     */
    public static void restoreViewTransformation(View view) throws IllegalStateException {
        Object tag = view.getTag();

        if (tag == null) {
            return;
        }

        if (!(tag instanceof float[])) {
            throw new IllegalStateException();
        }

        float[] values = (float[]) tag;
        view.setTranslationX(values[TRANSLATION_X]);
        view.setTranslationY(values[TRANSLATION_Y]);
        view.setScaleX(values[SCALE_X]);
        view.setScaleY(values[SCALE_Y]);
        view.setRotation(values[ROTATION]);
        view.setAlpha(values[ALPHA]);
    }

    /**
     * Executes {@link java.lang.Runnable} right before global draw event, which is convenient if you
     * want to be sure that provided {@link android.view.View} correctly calculated it's size.
     *
     * @param view
     * @param runnable
     */
    public static void postOnPreDraw(final View view, final Runnable runnable) {
        view.getViewTreeObserver().addOnPreDrawListener(new ViewTreeObserver.OnPreDrawListener() {
            @Override
            public boolean onPreDraw() {
                try {
                    runnable.run();
                    return true;
                } finally {
                    view.getViewTreeObserver().removeOnPreDrawListener(this);
                }
            }
        });
    }

    /**
     * @return visible rectangle of {@link android.view.View}
     */
    public static Rect getViewBounds(View view) {
        Rect bounds = new Rect();
        view.getGlobalVisibleRect(bounds);

        return bounds;
    }

    /**
     * Convenient alternative for {@link #fitRectToView(android.view.View, float, android.graphics.RectF)} which
     * creates rectangle from {@link android.graphics.Bitmap}
     */
    public static float fitBitmapToView(@Nullable Bitmap bitmap, @NotNull View view, float rotation) {
        if (bitmap == null) {
            return ViewUtils.SCALE_DEFAULT;
        }

        RectF imageBounds = new RectF(0f, 0f, bitmap.getWidth(), bitmap.getHeight());
        return fitRectToView(view, rotation, imageBounds);
    }

    /**
     * Convenient alternative for {@link #fitRectToView(android.view.View, float, android.graphics.RectF)} which
     * creates rectangle from intrinsic size of {@link android.graphics.drawable.Drawable}
     */
    public static float fitDrawableToView(@Nullable Drawable drawable, @NotNull View view, float rotation) {
        if (drawable == null) {
            return ViewUtils.SCALE_DEFAULT;
        }

        RectF drawableBounds = new RectF(
                0f, 0f,
                drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight()
        );

        return fitRectToView(view, rotation, drawableBounds);
    }

    /**
     * @param imageBounds {@link android.graphics.RectF} to fit
     * @param view        {@link android.view.View} in which rectangle should fit
     * @param rotation    rotation of {@link android.graphics.RectF} in degrees relatively to its center
     * @return new scale of rotated {@link android.graphics.RectF}
     */
    public static float fitRectToView(View view, float rotation, RectF imageBounds) {
        RectF viewBounds = new RectF(0f, 0f, view.getWidth(), view.getHeight());
        Matrix matrix = new Matrix();
        matrix.setRotate(rotation, imageBounds.centerX(), imageBounds.centerY());
        matrix.mapRect(imageBounds);
        matrix.reset();
        matrix.setRectToRect(imageBounds, viewBounds, Matrix.ScaleToFit.FILL);
        float[] m = new float[9];
        matrix.getValues(m);

        return Math.min(m[Matrix.MSCALE_X], m[Matrix.MSCALE_Y]);
    }

    /**
     * Implementation of {@link android.animation.Animator.AnimatorListener} that does nothing.
     * Used as base class for creating anonymous classes, which doesn't force you to override all the methods.
     * <p/>
     * Also, ensures that {@link #onAnimationFinished(android.animation.Animator)} will be invoked not more than once per animation (which, for instance, can be possible
     * on HTC One X).
     */
    public static class DefaultAnimationListener implements android.animation.Animator.AnimatorListener {

        private boolean ended = false;

        @Override
        public void onAnimationStart(android.animation.Animator animator) {
            ended = false;
        }

        @Override
        public final void onAnimationEnd(android.animation.Animator animator) {
            if (ended) {
                return;
            }

            onAnimationFinished(animator);
            ended = true;
        }

        /**
         * Implement this method instead of {@link #onAnimationEnd(android.animation.Animator)}
         */
        public void onAnimationFinished(android.animation.Animator animator) {
        }

        @Override
        public void onAnimationCancel(android.animation.Animator animator) {
        }

        @Override
        public void onAnimationRepeat(android.animation.Animator animator) {
        }
    }

    /**
     * Applies {@link View#LAYER_TYPE_HARDWARE} to provided {@link android.view.View} on animation start
     * and returns in to previous layer on animation finish.
     */
    public static class HardwareLayerAnimationListener extends DefaultAnimationListener {

        private final View view;
        private final int layerType;

        public HardwareLayerAnimationListener(View view) {
            this.view = view;
            layerType = view.getLayerType();
        }

        @Override
        public void onAnimationStart(Animator animator) {
            view.setLayerType(View.LAYER_TYPE_HARDWARE, null);
        }

        @Override
        public void onAnimationFinished(Animator animator) {
            view.setLayerType(layerType, null);
        }
    }
}
