package net.doo.snap.lib.snap.preview.zoom;

import android.animation.Animator;
import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.util.Property;
import android.view.GestureDetector;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;

import com.google.inject.Inject;
import com.ortiz.touch.TouchImageView;

import net.doo.snap.lib.R;
import net.doo.snap.lib.ui.events.BackPressedEvent;
import net.doo.snap.lib.ui.util.ViewUtils;

import roboguice.event.EventManager;
import roboguice.event.Observes;
import roboguice.fragment.RoboFragment;

/**
 * Displays {@link android.graphics.drawable.Drawable} with zooming functionality. Designed to be used as an
 * overlay on top of view being zoomed.
 * <p/>
 * Zooming workflow is controlled by set of following events:
 * - {@link net.doo.snap.lib.snap.preview.zoom.PrepareZoomEvent} provides initial information required to start zooming
 * - {@link net.doo.snap.lib.snap.preview.zoom.PreScaleChangedEvent} provides changes of scale of initial view during first pinch-to-zoom gesture
 * - {@link StartZoomingEvent} indicates that from now on this fragment is responsible for zooming workflow and will consume all touch events
 * - {@link net.doo.snap.lib.snap.preview.zoom.ZoomingFinishedEvent} fired by this fragment when zooming workflow should be finished. Doesn't consumes touch events from this moment.
 * <p/>
 * Doesn't save any state.
 */
public class ZoomedPreviewFragment extends RoboFragment implements View.OnTouchListener {

    private static final int MAX_DIM_ALPHA = 230;
    private static final float DEFAULT_SCALE = 1f;
    private static final float FINISH_SCALE_THRESHOLD = 1.3f;
    private static final Property<ZoomedPreviewFragment, Float> PRE_SCALE = Property.of(ZoomedPreviewFragment.class, Float.TYPE, "preScale");
    private static final Property<TouchImageView, Float> ZOOM = Property.of(TouchImageView.class, Float.TYPE, "zoom");

    @Inject
    private EventManager eventManager;

    private TouchImageView imageView;
    private ColorDrawable background;

    private boolean zooming = false;
    private boolean scaleCanceled = false;
    private float initialScale;
    private float preScale;
    private Object token;

    private ValueAnimator.AnimatorUpdateListener backgroundUpdater = new ValueAnimator.AnimatorUpdateListener() {
        @Override
        public void onAnimationUpdate(ValueAnimator valueAnimator) {
            updateBackground(imageView.getZoom());
        }
    };
    private int initialOffsetX;
    private int initialOffsetY;

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.scaled_preview_fragment, container, false);
        imageView = (TouchImageView) view.findViewById(R.id.image);
        imageView.setOnTouchListener(this);
        imageView.setOnDoubleTapListener(new GestureDetector.OnDoubleTapListener() {
            @Override
            public boolean onSingleTapConfirmed(MotionEvent event) {
                return true;
            }

            @Override
            public boolean onDoubleTap(MotionEvent event) {
                if (imageView.getZoom() > DEFAULT_SCALE) {
                    imageView.animateZoomTo(DEFAULT_SCALE);
                } else {
                    imageView.animateZoomTo(
                            imageView.getMaxZoom(),
                            event.getX(),
                            event.getY()
                    );
                }

                return true;
            }

            @Override
            public boolean onDoubleTapEvent(MotionEvent event) {
                return false;
            }
        });

        background = (ColorDrawable) getResources().getDrawable(R.color.scale_page_dim_color);
        background.setAlpha(0);
        view.setBackgroundDrawable(background);

        view.setLayerType(View.LAYER_TYPE_HARDWARE, null);

        return view;
    }

    @SuppressWarnings("unused")
    public void onPrepareZooming(@Observes PrepareZoomEvent event) {
        Drawable drawable = event.getDrawable();

        Matrix matrix = new Matrix();
        float[] m = new float[9];
        RectF image = new RectF(0f, 0f, drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight());
        RectF upscaled = new RectF(0f, 0f, getView().getWidth(), getView().getHeight());

        matrix.setRectToRect(image, upscaled, Matrix.ScaleToFit.START);
        matrix.mapRect(image);
        matrix.reset();

        Rect containerBounds = event.getInitialBounds();
        RectF original = new RectF(0f, 0f, containerBounds.width(), containerBounds.height());

        matrix.setRectToRect(image, original, Matrix.ScaleToFit.START);
        matrix.getValues(m);

        initialScale = Math.min(m[Matrix.MSCALE_X], m[Matrix.MSCALE_Y]);

        Rect upscaledContainer = ViewUtils.getViewBounds(getView());
        initialOffsetX = containerBounds.centerX() - upscaledContainer.centerX();
        initialOffsetY = containerBounds.centerY() - upscaledContainer.centerY();

        imageView.setTranslationX(initialOffsetX);
        imageView.setTranslationY(initialOffsetY);
        imageView.animate()
                .translationX(ViewUtils.TRANSLATION_DEFAULT)
                .translationY(ViewUtils.TRANSLATION_DEFAULT)
                .start();

        imageView.setImageDrawable(drawable);
        imageView.setZoom(DEFAULT_SCALE);
        imageView.setMinZoom(initialScale);
        setPreScale(initialScale);
        token = event.getToken();
    }

    @SuppressWarnings("unused")
    public void onPreScaleChanged(@Observes PreScaleChangedEvent preScaleChangedEvent) {
        final float scale = initialScale * preScaleChangedEvent.scale;

        setPreScale(scale);
    }

    @SuppressWarnings("unused")
    public void onZoomingStarted(@Observes StartZoomingEvent startZoomingEvent) {
        zooming = true;

        if (preScale > DEFAULT_SCALE) {
            imageView.setZoom(preScale);
            setPreScale(DEFAULT_SCALE);
        }

        updatePreScaleState();
    }

    @SuppressWarnings("unused")
    public void onQuickScale(@Observes QuickScaleEvent event) {
        zooming = true;
        resetPreScale();
    }

    private void updatePreScaleState() {
        if (preScale < Math.min(DEFAULT_SCALE, initialScale * FINISH_SCALE_THRESHOLD)) {
            cancelPreScale();
        } else if (preScale < DEFAULT_SCALE) {
            resetPreScale();
        }
    }

    private void resetPreScale() {
        ObjectAnimator.ofFloat(ZoomedPreviewFragment.this, PRE_SCALE, DEFAULT_SCALE).start();
    }

    private void cancelPreScale() {
        scaleCanceled = true;
        ObjectAnimator animator = ObjectAnimator.ofFloat(this, PRE_SCALE, initialScale);
        animator.addListener(new ViewUtils.DefaultAnimationListener() {
            @Override
            public void onAnimationFinished(Animator animator) {
                finish();
            }
        });
        animator.start();

        animateToInitialOffset();
    }

    /**
     * Changes scale transformation of zoomed {@link android.view.View}
     */
    public void setPreScale(float scale) {
        preScale = scale;
        imageView.setScaleX(scale);
        imageView.setScaleY(scale);

        updateBackground(scale);
    }

    /**
     * @return scale transformation of zoomed {@link android.view.View}
     */
    @SuppressWarnings("unused")
    public float getPreScale() {
        return preScale;
    }

    @Override
    public boolean onTouch(View view, MotionEvent event) {
        if (imageView.getDrawable() == null || !zooming || scaleCanceled) {
            return true;
        }

        final int action = event.getAction();
        final float zoom = imageView.getZoom();

        if (action == MotionEvent.ACTION_CANCEL || action == MotionEvent.ACTION_UP) {
            updateScaleState(zoom);
        }

        updateBackground(zoom);

        return false;
    }

    private void updateScaleState(float scale) {
        if (scale < Math.min(initialScale * FINISH_SCALE_THRESHOLD, DEFAULT_SCALE)) {
            cancelScale();
        } else if (scale < DEFAULT_SCALE) {
            resetScale();
        }
    }

    private void resetScale() {
        ObjectAnimator animator = ObjectAnimator.ofFloat(imageView, ZOOM, DEFAULT_SCALE);
        animator.addUpdateListener(backgroundUpdater);
        animator.start();
    }

    private void cancelScale() {
        scaleCanceled = true;
        ObjectAnimator animator = ObjectAnimator.ofFloat(imageView, ZOOM, initialScale);
        animator.addUpdateListener(backgroundUpdater);
        animator.addListener(new ViewUtils.DefaultAnimationListener() {
            @Override
            public void onAnimationFinished(Animator animator) {
                finish();
            }
        });
        animator.start();

        animateToInitialOffset();
    }

    private void finish() {
        if (!zooming) {
            //We're already finished
            return;
        }

        setPreScale(0f);
        imageView.setZoom(DEFAULT_SCALE);
        zooming = false;
        scaleCanceled = false;
        imageView.setImageDrawable(null);
        eventManager.fire(new ZoomingFinishedEvent(token));
        token = null;
    }

    private void animateToInitialOffset() {
        imageView.animate()
                .translationX(initialOffsetX)
                .translationY(initialOffsetY)
                .start();
    }

    private void updateBackground(float scale) {
        final float fraction = (scale - initialScale) / (DEFAULT_SCALE - initialScale);
        final float normalizedFraction = Math.max(Math.min(DEFAULT_SCALE, fraction), 0f);

        background.setAlpha((int) (MAX_DIM_ALPHA * normalizedFraction));
    }

    @SuppressWarnings("unused")
    public void onBackPressed(@Observes BackPressedEvent event) {
        if (zooming) {
            cancelScale();
        }
    }

}
