package net.doo.snap.lib.snap.camera;

import android.app.Activity;
import android.content.SharedPreferences;
import android.graphics.PointF;
import android.hardware.Camera;
import android.os.Bundle;
import android.os.Handler;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.DecelerateInterpolator;
import android.view.animation.Interpolator;
import android.view.animation.OvershootInterpolator;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.RelativeLayout.LayoutParams;

import com.commonsware.cwac.camera.CameraFragment;
import com.commonsware.cwac.camera.DeviceProfile;
import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.AnalyticsConst;
import net.doo.snap.lib.analytics.EasyTrackerWrapper;
import net.doo.snap.lib.detector.CameraDetectorListener;
import net.doo.snap.lib.sensor.SensorHelper;
import net.doo.snap.lib.snap.AutoSnappingDetectionHelper;
import net.doo.snap.lib.snap.event.PictureProcessingStatusChangedEvent;
import net.doo.snap.lib.snap.event.PictureTakenEvent;
import net.doo.snap.lib.snap.util.OrientationHandler;
import net.doo.snap.lib.snap.util.ScreenRotationChangedEvent;
import net.doo.snap.lib.snap.util.Utils;
import net.doo.snap.lib.ui.BarcodeFragment;
import net.doo.snap.lib.ui.Nexus4RebootFragment;
import net.doo.snap.lib.ui.SnappingFragment;
import net.doo.snap.lib.ui.util.ViewUtils;
import net.doo.snap.lib.ui.widget.ShutterDrawable;
import net.doo.snap.lib.util.CameraConfiguration;
import net.doo.snap.lib.util.State;
import net.doo.snap.lib.util.StateHolder;

import org.apache.commons.lang.BooleanUtils;

import java.util.List;

import roboguice.RoboGuice;
import roboguice.event.EventManager;
import roboguice.event.Observes;

/**
 * Displays camera preview with ability to take picture
 *
 * @see com.commonsware.cwac.camera.CameraFragment
 */
public class CameraPreviewFragment extends CameraFragment {
    private static final int CONTOUR_DETECTION_DELAY_MS = 2000;
    private static final int FLASH_KEEP_DELAY_MS = 4000;
    private static final int MANUAL_SNAPPING_DELAY_MS = 12000;

    private static final int PLACEHOLDER_FADE_OUT_DURATION_MILLIS = 700;
    private static final float TAKE_PICTURE_PRESSED_SCALE = 0.8f;
    private static final float TAKE_PICTURE_OVERSHOOT_TENSION = 8f;

    private static final String IS_IN_PREVIEW = "IS_IN_PREVIEW";
    private static final String IS_ACTIVE = "IS_ACTIVE";
    public static final String CAMERA_STARTED = "CAMERA_STARTED";

    @Inject
    private SnapCameraHost snapCameraHost;
    @Inject
    private EventManager eventManager;
    @Inject
    private CameraPreviewConfiguration cameraPreviewConfiguration;
    @Inject
    private OrientationHandler orientationHandler;
    @Inject
    private SharedPreferences preferences;
    @Inject
    private SensorHelper sensorHelper;
    @Inject
    private CameraConfiguration cameraConfiguration;

    private final Handler handler = new Handler();

    private final Runnable switchOffFlashRunnable = new Runnable() {
        @Override
        public void run() {
            flashToggle.setChecked(false);
        }
    };

    private final Runnable switchToManualModeRunnable = new Runnable() {
        @Override
        public void run() {
            if (isAutosnapEnabled()) {
                setAutoSnapEnabled(false);
            }
        }
    };

    private boolean cameraCanAnimate;

    private boolean isRefocusing = false;

    /**
     * State in which fragment is ready for snapping (camera is visible)
     */
    private final State READY = new State() {
        @Override
        public void onEnter() {
            if (!cameraCanAnimate) {
                cameraPlaceholder.animate()
                        .alpha(ViewUtils.ALPHA_TRANSPARENT)
                        .setDuration(PLACEHOLDER_FADE_OUT_DURATION_MILLIS)
                        .start();

                cameraPreview.setVisibility(View.VISIBLE);
            }

            takePicture.animate().alpha(ViewUtils.ALPHA_DEFAULT).start();
            flashToggle.animate().alpha(ViewUtils.ALPHA_DEFAULT).start();

            takePicture.setEnabled(true);
            flashToggle.setEnabled(true);

            cameraPreview.setPreviewCallback(snapCameraHost);
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (isAdded()
                            && cameraStateHolder.getCurrentState() == READY) {
                        snapCameraHost.setUseContourDetection(true);
                        cameraPreview.setDrawPolygon(true);
                    }
                }
            }, CONTOUR_DETECTION_DELAY_MS);

            orientationHandler.enable();

            setAutoSnapEnabled(true);
            handler.postDelayed(switchToManualModeRunnable, MANUAL_SNAPPING_DELAY_MS);

            Camera.Parameters params = cameraPreview.getCameraParameters();
            Utils.enableContinuousFocus(params);
            cameraPreview.setCameraParameters(params);

            cameraPreview.setOnTouchListener(new View.OnTouchListener() {
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    if (event.getAction() == MotionEvent.ACTION_DOWN) {
                        isRefocusing = true;
                        autoFocus();
                    }
                    return false;
                }
            });

            sensorHelper.registerSignificantMoveListener(cameraPreview);
        }

        @Override
        public void onLeave() {
            cameraPlaceholder.animate().cancel();
            takePicture.animate().cancel();
            flashToggle.animate().cancel();

            orientationHandler.disable();

            handler.removeCallbacks(switchToManualModeRunnable);

            cameraPreview.setOnTouchListener(null);
            sensorHelper.unregisterSignificantMoveListener(cameraPreview);
        }
    };

    /**
     * Preview state of fragment (current fragment is not selected in {@link net.doo.snap.lib.ui.widget.ViewPager})
     */
    private final State PREVIEW = new State() {
        @Override
        public void onEnter() {
            if (!cameraCanAnimate) {
                cameraPlaceholder.setAlpha(ViewUtils.ALPHA_DEFAULT);
                cameraPreview.setVisibility(View.GONE);
            }

            takePicture.setEnabled(false);
            flashToggle.setEnabled(false);

            takePicture.animate()
                    .alpha(ViewUtils.ALPHA_TRANSPARENT)
                    .scaleX(ViewUtils.SCALE_DEFAULT)
                    .scaleY(ViewUtils.SCALE_DEFAULT)
                    .start();

            flashToggle.animate().alpha(ViewUtils.ALPHA_TRANSPARENT).start();

            snapCameraHost.setUseContourDetection(false);
            cameraPreview.setDrawPolygon(false);
            cameraPreview.setPreviewCallback(null);
            cameraPreview.resetNotifications();

            handler.postDelayed(switchOffFlashRunnable, FLASH_KEEP_DELAY_MS);
        }

        @Override
        public void onLeave() {
            takePicture.animate().cancel();
            flashToggle.animate().cancel();

            handler.removeCallbacks(switchOffFlashRunnable);
        }
    };

    private State startState = PREVIEW;

    private SnapCameraView cameraPreview;
    private View cameraPlaceholder;
    private CheckBox flashToggle;
    private ImageView takePicture;
    private View progressView;

    private ShutterDrawable shutterDrawable;

    private boolean active = false;

    private StateHolder cameraStateHolder = new StateHolder();
    private float openedProgress;

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);
        cameraCanAnimate = DeviceProfile.getInstance(activity).useTextureView();
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        RoboGuice.getInjector(getActivity()).injectMembersWithoutViews(this);
        snapCameraHost.setCameraPreviewFragment(this);
        setHost(snapCameraHost);
    }

    /**
     * @return currently detected polygon
     * @see net.doo.snap.lib.detector.ContourDetector#detect
     */
    public List<PointF> getPolygon() {
        if (cameraPreview != null) {
            return cameraPreview.getPolygon();
        }

        return null;
    }

    /**
     * Display rotation depending on camera orientation
     * used to rotate polygon correctly
     *
     * @return degrees of rotation (0, 90, 180, 270)
     */
    public int getRotation() {
        int rotation;
        if (cameraPreview != null) {
            rotation = cameraPreview.getDisplayOrientation();
        } else {
            rotation = getActivity().getWindow().getWindowManager().getDefaultDisplay().getRotation();
        }
        return rotation;
    }

    private void setFlashEnabled(boolean enabled) {
        cameraPreviewConfiguration.setFlashEnabled(enabled);
        Camera.Parameters parameters = cameraPreview.getCameraParameters();
        if (Utils.isFlashSupported(parameters)) {
            parameters.setFlashMode(enabled ? Camera.Parameters.FLASH_MODE_TORCH : Camera.Parameters.FLASH_MODE_OFF);
            cameraPreview.setCameraParameters(parameters);
        }
    }

    private void setAutoSnapEnabled(boolean enabled) {
        cameraPreviewConfiguration.setAutoSnapEnabled(enabled);
        cameraPreview.setAutoSnapEnabled(enabled);

        if (enabled) {
            shutterDrawable.startAnimation();
        } else {
            shutterDrawable.stopAnimation();
        }

        EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_BUTTON_PRESS, "auto_snap_toggle", (long) BooleanUtils.toInteger(enabled));
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, final Bundle savedInstanceState) {
        RelativeLayout view = (RelativeLayout) inflater.inflate(R.layout.camera_preview_fragment, null);

        AutoSnappingDetectionHelper helper = new AutoSnappingDetectionHelper(getActivity(), view);

        cameraPreview = new SnapCameraView(getActivity(), helper, cameraConfiguration);
        cameraPreview.setHost(getHost());
        setCameraView(cameraPreview);
        LayoutParams params = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
        view.addView(cameraPreview, 0, params);

        takePicture = (ImageView) view.findViewById(R.id.take_picture);
        takePicture.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                autoFocus();
            }
        });
        takePicture.setOnTouchListener(new View.OnTouchListener() {

            private final Interpolator downInterpolator = new DecelerateInterpolator();
            private final Interpolator upInterpolator = new OvershootInterpolator(TAKE_PICTURE_OVERSHOOT_TENSION);

            @Override
            public boolean onTouch(View v, MotionEvent event) {
                switch (event.getAction()) {
                    case MotionEvent.ACTION_DOWN:
                        takePicture.animate()
                                .scaleX(TAKE_PICTURE_PRESSED_SCALE)
                                .scaleY(TAKE_PICTURE_PRESSED_SCALE)
                                .setInterpolator(downInterpolator)
                                .start();
                        break;
                    case MotionEvent.ACTION_CANCEL:
                    case MotionEvent.ACTION_UP:
                        takePicture.animate()
                                .scaleX(ViewUtils.SCALE_DEFAULT)
                                .scaleY(ViewUtils.SCALE_DEFAULT)
                                .setInterpolator(upInterpolator)
                                .start();
                        break;
                }

                return false;
            }
        });

        shutterDrawable = new ShutterDrawable(getResources());
        takePicture.setImageDrawable(shutterDrawable);

        flashToggle = (CheckBox) view.findViewById(R.id.flash);
        flashToggle.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton compoundButton, boolean checked) {
                setFlashEnabled(checked);
            }
        });

        progressView = view.findViewById(R.id.progress);

        cameraPlaceholder = view.findViewById(R.id.placeholder);

        if (!cameraCanAnimate) {
            cameraPlaceholder.setVisibility(View.VISIBLE);
        }

        ViewUtils.postOnPreDraw(cameraPreview, new Runnable() {
            @Override
            public void run() {
                restoreState(savedInstanceState);
            }
        });

        return view;
    }

    private void restoreState(Bundle state) {
        if (state == null) {
            cameraStateHolder.updateState(startState);
            return;
        }

        active = state.getBoolean(IS_ACTIVE, false);

        if (state.getBoolean(IS_IN_PREVIEW)) {
            cameraStateHolder.updateState(PREVIEW);
        } else {
            cameraStateHolder.updateState(READY);
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putBoolean(IS_IN_PREVIEW, cameraStateHolder.getCurrentState() == PREVIEW);
        outState.putBoolean(IS_ACTIVE, active);
    }

    @Override
    public void onResume() {
        super.onResume();
        if (preferences.getBoolean(CAMERA_STARTED, false)) {
            Nexus4RebootFragment.newInstance().showIfNexus4(getFragmentManager());
        }
        preferences.edit().putBoolean(CAMERA_STARTED, true).commit();
        flashToggle.setChecked(cameraPreviewConfiguration.isFlashEnabled());

        if (cameraStateHolder.getCurrentState() == READY) {
            orientationHandler.enable();
        }

        setupExceptionHandler();
    }

    private void setupExceptionHandler() {
        Thread.currentThread().setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
            @Override
            public void uncaughtException(Thread thread, Throwable ex) {
                if (cameraPreview != null) {
                    cameraPreview.onPause();
                }

                Thread.UncaughtExceptionHandler defaultHandler = Thread.getDefaultUncaughtExceptionHandler();
                if (defaultHandler != null) {
                    defaultHandler.uncaughtException(thread, ex);
                }
            }
        });
    }

    @Override
    public void onPause() {
        super.onPause();
        orientationHandler.disable();
        preferences.edit().putBoolean(CAMERA_STARTED, false).commit();

        resetExceptionHandler();
    }

    private void resetExceptionHandler() {
        Thread.currentThread().setUncaughtExceptionHandler(Thread.getDefaultUncaughtExceptionHandler());
    }

    @Override
    public void onDestroyView() {
        cameraStateHolder.updateState(State.DEFAULT);

        super.onDestroyView();
    }

    /**
     * Changes camera preview mode. In preview mode snapping controls are not visible, contour detection is not active
     * and preview buffers are not set. In normal code snapping controls are available, contour detection is working and preview
     * buffers are assigned.
     * <p/>
     * Note that this method should not be used in combination with {@link #setOpenProgress(float)} or {@link #setCameraActive(boolean)},
     * since one replaces effect of another.
     *
     * @param previewMode {@code true} for preview mode. {@code false} for normal mode.
     */
    public void setPreviewMode(boolean previewMode) {
        active = !previewMode;
        startState = previewMode
                ? PREVIEW
                : READY;
        cameraStateHolder.updateState(startState);
    }

    /**
     * @return {@code true} if {@link CameraPreviewFragment} is ready to snap (UI is shown and image processing
     * is turned on). {@code false} otherwise.
     */
    public boolean isReady() {
        return cameraStateHolder.getCurrentState() == READY;
    }

    /**
     * Notifies {@link CameraPreviewFragment} about it state in {@link net.doo.snap.lib.ui.widget.ViewPager}.
     *
     * @param progress value from 0.0 to 1.0 (from fully closed to fully opened state)
     */
    public void setOpenProgress(float progress) {
        openedProgress = progress;
        if (!active || getView() == null) {
            return;
        }

        cameraStateHolder.updateState(
                progress == SnappingFragment.PAGE_OPENED_PROGRESS
                        ? READY
                        : PREVIEW
        );
    }

    /**
     * sets if camera has flash support
     *
     * @param supported
     */
    public void setFlashButtonVisible(boolean supported) {
        flashToggle.setVisibility(supported ? View.VISIBLE : View.GONE);
    }

    /**
     * @return {@code true} if flash is enabled in the control panel, {@code false} otherwise. By default {@code false}
     */
    public boolean isFlashEnabled() {
        return flashToggle.isChecked();
    }

    /**
     * @return {@link net.doo.snap.lib.detector.CameraDetectorListener} which is responsible for handling
     * detection results.
     */
    public CameraDetectorListener getDetectorListener() {
        return cameraPreview;
    }

    /**
     * Invoked when picture was taken
     */
    public void onPictureTaken(byte[] image, int imageOrientation) {
        eventManager.fire(new PictureTakenEvent(image, imageOrientation));
    }

    /**
     * Invoked when processing of picture was started
     */
    public void onStartPictureProcessing() {
        eventManager.fire(new PictureProcessingStatusChangedEvent(true));

        handler.post(new Runnable() {
            @Override
            public void run() {
                if (!isAdded()) {
                    return;
                }

                progressView.setVisibility(View.VISIBLE);
                cameraStateHolder.updateState(PREVIEW);
                cameraPreview.setDrawPolygon(true);
            }
        });
    }

    /**
     * Invoked when processing of picture was finished
     */
    public void onFinishPictureProcessing() {
        eventManager.fire(new PictureProcessingStatusChangedEvent(false));

        handler.post(new Runnable() {
            @Override
            public void run() {
                if (!isAdded()) {
                    return;
                }

                progressView.setVisibility(View.GONE);
            }
        });
    }

    /**
     * Starts camera preview.
     *
     * @see com.commonsware.cwac.camera.CameraView#startPreview()
     */
    public void startCameraPreview() {
        cameraPreview.startPreview();
    }

    /**
     * Enables or disables workflow of this fragment.
     * When enabled, camera controls are shown and image processing turned on.
     * When disabled, camera controls are hidden and image processing turned off.
     */
    public void setCameraActive(boolean active) {
        if (!isAdded()) {
            return;
        }
        this.active = active;

        if (this.active) {
            setOpenProgress(openedProgress);
        } else {
            cameraStateHolder.updateState(PREVIEW);
        }
    }

    @SuppressWarnings("unused")
    public void onScreenRotationChanged(@Observes ScreenRotationChangedEvent event) {
        if (!isAdded()) {
            return;
        }

        final int rotation = -event.getRotation();
        flashToggle.animate().rotation(rotation).start();
    }

    /**
     * @return if autoSnap enabled or not.
     * If not then we treat it as if manual mode is enabled
     */
    public boolean isAutosnapEnabled() {
        return cameraPreviewConfiguration.isAutoSnapEnabled();
    }

    /**
     * @return true if camera is refocusing using screen tap
     */
    public boolean isRefocusing() {
        return isRefocusing;
    }

    /**
     * Called when refocusing is finished to disable isRefocusing()
     */
    public void finishRefocusing() {
        isRefocusing = false;
    }

    /**
     * Shows {@link net.doo.snap.lib.ui.BarcodeFragment} for barcode content handling
     */
    public void onBarcodeDetected() {
        setPreviewMode(true);
        getFragmentManager()
                .beginTransaction()
                .setCustomAnimations(android.R.anim.fade_in, android.R.anim.fade_out)
                .add(R.id.root, new BarcodeFragment(), BarcodeFragment.BARCODE_FRAGMENT_TAG)
                .addToBackStack(BarcodeFragment.BARCODE_FRAGMENT_TAG)
                .commitAllowingStateLoss();
    }
}
