package net.doo.snap.lib.snap.edit;

import android.animation.Animator;
import android.app.Activity;
import android.app.Dialog;
import android.graphics.Bitmap;
import android.graphics.Rect;
import android.graphics.drawable.BitmapDrawable;
import android.os.Bundle;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.Loader;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.ImageView;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.persistence.OptimizationType;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageStoreStrategy;
import net.doo.snap.lib.persistence.RotationType;
import net.doo.snap.lib.snap.event.PageOptimizationTypeChangedEvent;
import net.doo.snap.lib.ui.util.TransformableDrawable;
import net.doo.snap.lib.ui.util.ViewUtils;
import net.doo.snap.lib.util.bitmap.PageBitmapLoader;
import net.doo.snap.lib.util.log.DebugLog;

import java.io.IOException;

import roboguice.event.EventManager;
import roboguice.fragment.RoboDialogFragment;

/**
 * Shows preview of available {@link net.doo.snap.lib.persistence.OptimizationType}s and lets user
 * choose one. Fires {@link net.doo.snap.lib.snap.event.PageOptimizationTypeChangedEvent} on selection.
 * <p/>
 * Tied with {@link net.doo.snap.lib.ui.SnappingFragment} UI, so close-window button located exactly at the same
 * place as button, which opened this fragment.
 */
public class ChangeFilterFragment extends RoboDialogFragment implements LoaderManager.LoaderCallbacks<Bitmap> {

    private static final String ARG_PAGE = "ARG_PAGE";
    private static final String ARG_FILTER_BUTTON_BOUNDS = "ARG_FILTER_BUTTON_BOUNDS";
    private static final String ARG_FILTER_BUTTON_ID = "ARG_FILTER_BUTTON_ID";
    private static final String LOADER_IMAGE_PATH = "LOADER_IMAGE_PATH";
    private static final String ARG_VIEW_PAGER_ID = "ARG_VIEW_PAGER_ID";

    @Inject
    private PageStoreStrategy pageStoreStrategy;
    @Inject
    private EventManager eventManager;

    private Page page;
    private Rect filterButtonBounds;
    private int filterButtonId;
    private int viewPagerId;

    /**
     * @param page         {@link net.doo.snap.lib.persistence.Page} for which filter should be applied
     * @param filterButton {@link android.view.View} that was used to open this fragment
     * @param viewPagerId  id of {@link net.doo.snap.lib.ui.widget.ViewPager} which will be animated during appearance of dialog
     * @return new instance of {@link net.doo.snap.lib.snap.edit.ChangeFilterFragment}
     */
    public static ChangeFilterFragment newInstance(Page page, View filterButton, int viewPagerId) {
        ChangeFilterFragment fragment = new ChangeFilterFragment();

        Bundle args = new Bundle();
        args.putParcelable(ARG_PAGE, page);
        args.putParcelable(ARG_FILTER_BUTTON_BOUNDS, ViewUtils.getViewBounds(filterButton));
        args.putInt(ARG_FILTER_BUTTON_ID, filterButton.getId());
        args.putInt(ARG_VIEW_PAGER_ID, viewPagerId);
        fragment.setArguments(args);

        return fragment;
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        Bundle arguments = getArguments();
        page = arguments.getParcelable(ARG_PAGE);
        filterButtonBounds = arguments.getParcelable(ARG_FILTER_BUTTON_BOUNDS);
        filterButtonId = arguments.getInt(ARG_FILTER_BUTTON_ID);
        viewPagerId = arguments.getInt(ARG_VIEW_PAGER_ID);
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        setStyle(STYLE_NO_FRAME, android.R.style.Theme_DeviceDefault_NoActionBar);
        super.onCreate(savedInstanceState);

        runLoaders();
    }

    private void runLoaders() {
        try {
            for (OptimizationType optimizationType : OptimizationType.values()) {
                final String path = pageStoreStrategy.getFilteredPreviewFile(page.getId(), optimizationType).getPath();

                Bundle args = new Bundle();
                args.putString(LOADER_IMAGE_PATH, path);

                getLoaderManager().initLoader(optimizationType.ordinal(), args, this);
            }
        } catch (IOException e) {
            DebugLog.logException(e);
        }
    }

    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        Dialog dialog = super.onCreateDialog(savedInstanceState);

        Window window = dialog.getWindow();
        window.setBackgroundDrawableResource(android.R.color.transparent);
        window.clearFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);

        return dialog;
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        super.onCreateView(inflater, container, savedInstanceState);

        final View root = inflater.inflate(R.layout.change_filter_fragment, container, false);

        ViewUtils.postOnPreDraw(root, new Runnable() {
            @Override
            public void run() {
                initCloseButton(root);
            }
        });

        root.setAlpha(ViewUtils.ALPHA_TRANSPARENT);
        root
                .animate()
                .alpha(ViewUtils.ALPHA_DEFAULT)
                .setListener(new ViewUtils.HardwareLayerAnimationListener(root));

        return root;
    }

    private void initCloseButton(View root) {
        View closeButton = root.findViewById(R.id.close);

        closeButton.setTranslationX(filterButtonBounds.centerX() - closeButton.getWidth() / 2);
        closeButton.setTranslationY(filterButtonBounds.top - closeButton.getHeight() / 2);

        closeButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                animateDismiss();
            }
        });
    }

    @Override
    public void onViewCreated(View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        ViewUtils.postOnPreDraw(view, new Runnable() {
            @Override
            public void run() {
                initPreviews();
            }
        });
    }

    private void initPreviews() {
        int[] position = new int[2];

        for (final OptimizationType optimizationType : OptimizationType.values()) {
            ImageView imageView = getImageViewForOptimizationType(optimizationType);

            imageView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    if (page.getOptimizationType() != optimizationType) {
                        eventManager.fire(new PageOptimizationTypeChangedEvent(page, optimizationType));
                    }

                    animateDismiss();
                }
            });

            final int startX = filterButtonBounds.centerX();
            final int startY = filterButtonBounds.top;

            imageView.getLocationInWindow(position);

            imageView.setTranslationX(startX - position[0]);
            imageView.setTranslationY(startY - position[1]);
            imageView.setScaleX(0.2f);
            imageView.setScaleY(0.2f);
            imageView.setAlpha(0.3f);
        }
    }

    private void animateDismiss() {
        getView()
                .animate()
                .alpha(ViewUtils.ALPHA_TRANSPARENT)
                .setListener(new ViewUtils.HardwareLayerAnimationListener(getView()) {
                    @Override
                    public void onAnimationFinished(Animator animator) {
                        if (!isAdded()) {
                            return;
                        }

                        dismissAllowingStateLoss();
                    }
                });

        Activity activity = getActivity();
        activity.findViewById(filterButtonId).animate().alpha(ViewUtils.ALPHA_DEFAULT);

        final View pager = activity.findViewById(viewPagerId);
        pager.findViewById(viewPagerId)
                .animate()
                .alpha(ViewUtils.ALPHA_DEFAULT)
                .setListener(new ViewUtils.HardwareLayerAnimationListener(pager));
    }

    @Override
    public void onResume() {
        super.onResume();

        Activity activity = getActivity();
        activity.findViewById(filterButtonId).animate().alpha(ViewUtils.ALPHA_TRANSPARENT);

        final View pager = activity.findViewById(viewPagerId);
        pager
                .animate()
                .alpha(ViewUtils.ALPHA_TRANSPARENT)
                .setListener(new ViewUtils.HardwareLayerAnimationListener(pager));
    }

    @Override
    public void onPause() {
        super.onPause();

        Activity activity = getActivity();
        activity.findViewById(filterButtonId).setAlpha(ViewUtils.ALPHA_DEFAULT);
        activity.findViewById(viewPagerId).setAlpha(ViewUtils.ALPHA_DEFAULT);

        dismissAllowingStateLoss();
    }

    @Override
    public Loader<Bitmap> onCreateLoader(int id, Bundle args) {
        String path = args.getString(LOADER_IMAGE_PATH);

        return new PageBitmapLoader(
                getActivity(),
                path,
                null,
                null
        );
    }

    @Override
    public void onLoadFinished(Loader<Bitmap> loader, final Bitmap data) {
        OptimizationType optimizationType = OptimizationType.values()[loader.getId()];
        final ImageView imageView = getImageViewForOptimizationType(optimizationType);
        imageView.setImageBitmap(data);

        ViewUtils.postOnPreDraw(imageView, new Runnable() {
            @Override
            public void run() {
                applyResultAndStartAnimation(data, imageView);
            }
        });
    }

    private void applyResultAndStartAnimation(Bitmap data, ImageView imageView) {
        TransformableDrawable transformableDrawable = new TransformableDrawable(new BitmapDrawable(getResources(), data));
        transformableDrawable.setRotation(page.getRotationType().getDegrees());
        transformableDrawable.setScale(calculateRotationScale(imageView, data, page.getRotationType()));

        imageView.setImageDrawable(transformableDrawable);

        imageView.animate()
                .alpha(ViewUtils.ALPHA_DEFAULT)
                .scaleX(ViewUtils.SCALE_DEFAULT)
                .scaleY(ViewUtils.SCALE_DEFAULT)
                .translationX(ViewUtils.TRANSLATION_DEFAULT)
                .translationY(ViewUtils.TRANSLATION_DEFAULT)
                .setListener(new ViewUtils.HardwareLayerAnimationListener(imageView));
    }

    @Override
    public void onLoaderReset(Loader<Bitmap> loader) {
    }

    private float calculateRotationScale(ImageView imageView, Bitmap bitmap, RotationType rotationType) {
        if (bitmap == null) {
            DebugLog.d("No rotation scale - bitmap is null");
            return ViewUtils.SCALE_DEFAULT;
        }

        final float availableWidth = imageView.getWidth();
        final float availableHeight = imageView.getHeight();

        if (rotationType == RotationType.ROTATION_0
                || rotationType == RotationType.ROTATION_180
                || rotationType == RotationType.ROTATION_360) {
            return Math.min(
                    availableWidth / bitmap.getWidth(),
                    availableHeight / bitmap.getHeight()
            );
        }

        return Math.min(
                availableWidth / bitmap.getHeight(),
                availableHeight / bitmap.getWidth()
        );
    }

    private ImageView getImageViewForOptimizationType(OptimizationType optimizationType) {
        View view;
        switch (optimizationType) {
            case NONE:
                view = getView().findViewById(R.id.filter_none);
                break;
            case COLOR_ENHANCED:
                view = getView().findViewById(R.id.filter_color);
                break;
            case GRAYSCALE:
                view = getView().findViewById(R.id.filter_grayscale);
                break;
            case BLACK_AND_WHITE:
                view = getView().findViewById(R.id.filter_black_and_white);
                break;
            default:
                throw new UnsupportedOperationException();
        }

        return (ImageView) view;
    }
}
