package net.doo.snap.lib.util.billing;

import android.app.Activity;
import android.app.PendingIntent;
import android.content.*;
import android.content.IntentSender.SendIntentException;
import android.content.pm.ResolveInfo;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.preference.PreferenceManager;
import android.text.TextUtils;
import com.android.vending.billing.IInAppBillingService;
import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.billing.SecurityManager;
import net.doo.snap.lib.util.log.DebugLog;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;


/**
 * Provides convenience methods for in-app billing. You can create one instance of this
 * class for your application and use it to process in-app billing operations.
 * It provides synchronous (blocking) methods for
 * many common in-app billing operations, as well as automatic signature
 * verification.
 * <p/>
 * After instantiating, you must perform setup in order to start using the object.
 * To perform setup, call the {@link #startSetup} method and provide a listener;
 * that listener will be notified when setup is complete, after which (and not before)
 * you may call other methods.
 * <p/>
 * After setup is complete, you will typically want to request an inventory of owned
 * items and subscriptions. See {@link #queryInventory}
 * and related methods.
 * <p/>
 * When you are done with this object, don't forget to call {@link #dispose}
 * to ensure proper cleanup. This object holds a binding to the in-app billing
 * service, which will leak unless you dispose of it correctly. If you created
 * the object on an Activity's onCreate method, then the recommended
 * place to dispose of it is the Activity's onDestroy method.
 * <p/>
 * A note about threading: When using this object from a background thread, you may
 * call the blocking versions of methods.
 *
 * @author Bruno Oliveira (Google)
 */
public class IabHelper {
    private static final int BILLING_API_VERSION = 3;

    // Billing response codes
    public static final int BILLING_RESPONSE_RESULT_OK = 0;
    public static final int BILLING_RESPONSE_RESULT_USER_CANCELED = 1;
    public static final int BILLING_RESPONSE_RESULT_BILLING_UNAVAILABLE = 3;
    public static final int BILLING_RESPONSE_RESULT_ITEM_UNAVAILABLE = 4;
    public static final int BILLING_RESPONSE_RESULT_DEVELOPER_ERROR = 5;
    public static final int BILLING_RESPONSE_RESULT_ERROR = 6;
    public static final int BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED = 7;
    public static final int BILLING_RESPONSE_RESULT_ITEM_NOT_OWNED = 8;

    // IAB Helper error codes
    public static final int IABHELPER_ERROR_BASE = -1000;
    public static final int IABHELPER_REMOTE_EXCEPTION = -1001;
    public static final int IABHELPER_BAD_RESPONSE = -1002;
    public static final int IABHELPER_VERIFICATION_FAILED = -1003;
    public static final int IABHELPER_SEND_INTENT_FAILED = -1004;
    public static final int IABHELPER_USER_CANCELLED = -1005;
    public static final int IABHELPER_UNKNOWN_PURCHASE_RESPONSE = -1006;
    public static final int IABHELPER_MISSING_TOKEN = -1007;
    public static final int IABHELPER_UNKNOWN_ERROR = -1008;
    public static final int IABHELPER_SUBSCRIPTIONS_NOT_AVAILABLE = -1009;
    public static final int IABHELPER_INVALID_CONSUMPTION = -1010;

    // Keys for the responses from InAppBillingService
    public static final String RESPONSE_CODE = "RESPONSE_CODE";
    public static final String RESPONSE_GET_SKU_DETAILS_LIST = "DETAILS_LIST";
    public static final String RESPONSE_BUY_INTENT = "BUY_INTENT";
    public static final String RESPONSE_INAPP_PURCHASE_DATA = "INAPP_PURCHASE_DATA";
    public static final String RESPONSE_INAPP_SIGNATURE = "INAPP_DATA_SIGNATURE";
    public static final String RESPONSE_INAPP_ITEM_LIST = "INAPP_PURCHASE_ITEM_LIST";
    public static final String RESPONSE_INAPP_PURCHASE_DATA_LIST = "INAPP_PURCHASE_DATA_LIST";
    public static final String RESPONSE_INAPP_SIGNATURE_LIST = "INAPP_DATA_SIGNATURE_LIST";
    public static final String INAPP_CONTINUATION_TOKEN = "INAPP_CONTINUATION_TOKEN";

    public static final String ITEM_TYPE_INAPP = "inapp";
    public static final String ITEM_TYPE_SUBS = "subs";

    // some fields on the getSkuDetails response bundle
    public static final String GET_SKU_DETAILS_ITEM_LIST = "ITEM_ID_LIST";
    public static final String GET_SKU_DETAILS_ITEM_TYPE_LIST = "ITEM_TYPE_LIST";

    private final SecurityManager securityManager;
    private final SharedPreferences preferences;

    // Is setup done?
    private AtomicBoolean setupDone = new AtomicBoolean(false);

    // Has this object been disposed of? (If so, we should ignore callbacks, etc)
    private AtomicBoolean disposed = new AtomicBoolean(false);

    // Are subscriptions supported?
    private boolean subscriptionsSupported = false;

    // Is an asynchronous operation in progress?
    // (only one at a time can be in progress)
    private AtomicBoolean asyncInProgress = new AtomicBoolean(false);

    // Context we were passed during initialization
    private Context context;

    // Connection to the service
    private IInAppBillingService service;
    private ServiceConnection serviceConn;

    // The request code used to launch purchase flow
    private int requestCode;

    // The item type of the current purchase flow
    private String purchasingItemType;

    // The listener registered on launchPurchaseFlow, which we have to call back when
    // the purchase finishes
    private OnIabPurchaseFinishedListener purchaseListener;

    /**
     * Callback for setup process. This listener's {@link #onIabSetupFinished} method is called
     * when the setup process is complete.
     */
    public interface OnIabSetupFinishedListener {
        /**
         * Called to notify that setup is complete.
         *
         * @param result The result of the setup process.
         */
        public void onIabSetupFinished(IabResult result);
    }

    /**
     * Callback that notifies when a purchase is finished.
     */
    public interface OnIabPurchaseFinishedListener {
        /**
         * Called to notify that an in-app purchase finished. If the purchase was successful,
         * then the sku parameter specifies which item was purchased. If the purchase failed,
         * the sku and extraData parameters may or may not be null, depending on how far the purchase
         * process went.
         *
         * @param result The result of the purchase.
         * @param info   The purchase information (null if purchase failed)
         */
        public void onIabPurchaseFinished(IabResult result, Purchase info);
    }

    /**
     * Creates an instance. After creation, it will not yet be ready to use. You must perform
     * setup by calling {@link #startSetup} and wait for setup to complete. This constructor does not
     * block and is safe to call from a UI thread.
     *
     * @param ctx             Your application or Activity context. Needed to bind to the in-app billing service.
     */
    public IabHelper(Context ctx) {
        context = ctx.getApplicationContext();
        securityManager = new SecurityManager();
        preferences = PreferenceManager.getDefaultSharedPreferences(ctx);
        logDebug("IAB helper created.");
    }

    /**
     * Starts the setup process. This will start up the setup process asynchronously.
     * You will be notified through the listener when the setup process is complete.
     * This method is safe to call from a UI thread.
     *
     * @param listener The listener to notify when the setup process is complete.
     */
    public void startSetup(final OnIabSetupFinishedListener listener) {
        // If already set up, can't do it again.
        ensureNotDisposed();
        if (setupDone.get()) {
            throw new IllegalStateException("IAB helper is already set up.");
        }

        // Connection to IAB service
        logDebug("Starting in-app billing setup.");
        serviceConn = new ServiceConnection() {
            @Override
            public void onServiceDisconnected(ComponentName name) {
                logDebug("Billing service disconnected.");
                service = null;
            }

            @Override
            public void onServiceConnected(ComponentName name, IBinder service) {
                if (disposed.get()) {
                    return;
                }
                logDebug("Billing service connected.");
                IabHelper.this.service = IInAppBillingService.Stub.asInterface(service);
                String packageName = context.getPackageName();
                try {
                    logDebug("Checking for in-app billing 3 support.");

                    // check for in-app billing v3 support
                    int response = IabHelper.this.service.isBillingSupported(BILLING_API_VERSION, packageName, ITEM_TYPE_INAPP);
                    if (response != BILLING_RESPONSE_RESULT_OK) {
                        if (listener != null) {
                            listener.onIabSetupFinished(new IabResult(response,
                                    "Error checking for billing v3 support."));
                        }

                        // if in-app purchases aren't supported, neither are subscriptions.
                        subscriptionsSupported = false;
                        return;
                    }
                    logDebug("In-app billing version 3 supported for " + packageName);

                    // check for v3 subscriptions support
                    response = IabHelper.this.service.isBillingSupported(BILLING_API_VERSION, packageName, ITEM_TYPE_SUBS);
                    if (response == BILLING_RESPONSE_RESULT_OK) {
                        logDebug("Subscriptions AVAILABLE.");
                        subscriptionsSupported = true;
                    } else {
                        logDebug("Subscriptions NOT AVAILABLE. Response: " + response);
                    }

                    setupDone.set(true);
                } catch (RemoteException e) {
                    if (listener != null) {
                        listener.onIabSetupFinished(new IabResult(IABHELPER_REMOTE_EXCEPTION,
                                "RemoteException while setting up in-app billing."));
                    }
                    DebugLog.logException(e);
                    return;
                }

                if (listener != null) {
                    listener.onIabSetupFinished(new IabResult(BILLING_RESPONSE_RESULT_OK, "Setup successful."));
                }
            }
        };

        Intent serviceIntent = new Intent("com.android.vending.billing.InAppBillingService.BIND");
        serviceIntent.setPackage("com.android.vending");
        List<ResolveInfo> resolveInfos = context.getPackageManager().queryIntentServices(serviceIntent, 0);
        if (resolveInfos != null && !resolveInfos.isEmpty()) {
            // service available to handle that Intent
            context.bindService(serviceIntent, serviceConn, Context.BIND_AUTO_CREATE);
        } else {
            serviceConn = null;
            // no service available to handle that Intent
            if (listener != null) {
                listener.onIabSetupFinished(
                        new IabResult(BILLING_RESPONSE_RESULT_BILLING_UNAVAILABLE,
                                "Billing service unavailable on device."));
            }
        }
    }

    /**
     * Dispose of object, releasing resources. It's very important to call this
     * method when you are done with this object. It will release any resources
     * used by it such as service connections. Naturally, once the object is
     * disposed of, it can't be used again.
     */
    public void dispose() {
        logDebug("Disposing.");
        setupDone.set(false);
        if (serviceConn != null) {
            logDebug("Unbinding from service.");
            if (context != null) {
                context.unbindService(serviceConn);
            }
        }
        disposed.set(true);
        context = null;
        serviceConn = null;
        service = null;
        purchaseListener = null;
    }

    private void ensureNotDisposed() {
        if (disposed.get()) {
            throw new IllegalStateException("IabHelper was disposed of, so it cannot be used.");
        }
    }

    /**
     * @return {@code true} if setup process was completed. {@code false} otherwise.
     */
    public boolean isSetupDone() {
        return setupDone.get();
    }

    /**
     * Returns whether subscriptions are supported.
     */
    public boolean subscriptionsSupported() {
        return subscriptionsSupported;
    }

    /**
     * Same as {@link #launchPurchaseFlow(android.app.Activity, String, String, int, net.doo.snap.lib.util.billing.IabHelper.OnIabPurchaseFinishedListener, String)}
     * @param act
     * @param sku
     * @param requestCode
     * @param listener
     */
    public void launchPurchaseFlow(Activity act, String sku, int requestCode, OnIabPurchaseFinishedListener listener) {
        String nonce = securityManager.generateNonce();
        preferences.edit().putString(PreferencesConstants.SKU_DEVELOPER_PAYLOAD_PREFIX + sku, nonce).commit();
        launchPurchaseFlow(act, sku, requestCode, listener, nonce);
    }

    /**
     * Same as {@link #launchPurchaseFlow(android.app.Activity, String, String, int, net.doo.snap.lib.util.billing.IabHelper.OnIabPurchaseFinishedListener, String)}
     * @param act
     * @param sku
     * @param requestCode
     * @param listener
     * @param extraData
     */
    public void launchPurchaseFlow(Activity act, String sku, int requestCode,
                                   OnIabPurchaseFinishedListener listener, String extraData) {
        launchPurchaseFlow(act, sku, ITEM_TYPE_INAPP, requestCode, listener, extraData);
    }

    /**
     * Same as {@link #launchPurchaseFlow(android.app.Activity, String, String, int, net.doo.snap.lib.util.billing.IabHelper.OnIabPurchaseFinishedListener, String)}
     * but initiate subscription flow
     * @param act
     * @param sku
     * @param requestCode
     * @param listener
     */
    public void launchSubscriptionPurchaseFlow(Activity act, String sku, int requestCode,
                                               OnIabPurchaseFinishedListener listener) {
        String nonce = securityManager.generateNonce();
        preferences.edit().putString(PreferencesConstants.SKU_DEVELOPER_PAYLOAD_PREFIX + sku, nonce).commit();
        launchSubscriptionPurchaseFlow(act, sku, requestCode, listener, nonce);
    }

    /**
     * Same as {@link #launchPurchaseFlow(android.app.Activity, String, String, int, net.doo.snap.lib.util.billing.IabHelper.OnIabPurchaseFinishedListener, String)}
     * but initiate subscription flow
     * @param act
     * @param sku
     * @param requestCode
     * @param listener
     * @param extraData
     */
    public void launchSubscriptionPurchaseFlow(Activity act, String sku, int requestCode,
                                               OnIabPurchaseFinishedListener listener, String extraData) {
        launchPurchaseFlow(act, sku, ITEM_TYPE_SUBS, requestCode, listener, extraData);
    }

    /**
     * Initiate the UI flow for an in-app purchase. Call this method to initiate an in-app purchase,
     * which will involve bringing up the Google Play screen. The calling activity will be paused while
     * the user interacts with Google Play, and the result will be delivered via the activity's
     * {@link android.app.Activity#onActivityResult} method, at which point you must call
     * this object's {@link #handleActivityResult} method to continue the purchase flow. This method
     * MUST be called from the UI thread of the Activity.
     *
     * @param act         The calling activity.
     * @param sku         The sku of the item to purchase.
     * @param itemType    indicates if it's a product or a subscription (ITEM_TYPE_INAPP or ITEM_TYPE_SUBS)
     * @param requestCode A request code (to differentiate from other responses --
     *                    as in {@link android.app.Activity#startActivityForResult}).
     * @param listener    The listener to notify when the purchase process finishes
     * @param extraData   Extra data (developer payload), which will be returned with the purchase data
     *                    when the purchase completes. This extra data will be permanently bound to that purchase
     *                    and will always be returned when the purchase is queried.
     */
    public void launchPurchaseFlow(Activity act, String sku, String itemType, int requestCode,
                                   OnIabPurchaseFinishedListener listener, String extraData) {
        ensureNotDisposed();
        ensureSetupDone();
        flagStartAsync();

        if (itemType.equals(ITEM_TYPE_SUBS) && !subscriptionsSupported) {
            flagEndAsync();
            deliverResult(listener, null, IABHELPER_SUBSCRIPTIONS_NOT_AVAILABLE,
                    "Subscriptions are not available.");
            return;
        }

        try {
            logDebug("Constructing buy intent for " + sku + ", item type: " + itemType);
            Bundle buyIntentBundle = service.getBuyIntent(BILLING_API_VERSION, context.getPackageName(), sku, itemType, extraData);
            int response = getResponseCodeFromBundle(buyIntentBundle);
            if (response != BILLING_RESPONSE_RESULT_OK) {
                logError("Unable to buy item, Error response: " + getResponseDesc(response));
                flagEndAsync();
                deliverResult(listener, null, response, "Unable to buy item");
                return;
            }

            PendingIntent pendingIntent = buyIntentBundle.getParcelable(RESPONSE_BUY_INTENT);
            logDebug("Launching buy intent for " + sku + ". Request code: " + requestCode);
            this.requestCode = requestCode;
            purchaseListener = listener;
            purchasingItemType = itemType;
            act.startIntentSenderForResult(pendingIntent.getIntentSender(),
                    requestCode, new Intent(),
                    0, 0, 0);
        } catch (SendIntentException e) {
            logError("SendIntentException while launching purchase flow for sku " + sku);
            DebugLog.logException(e);
            flagEndAsync();

            deliverResult(listener, null, IABHELPER_SEND_INTENT_FAILED, "Failed to send intent.");
        } catch (RemoteException e) {
            logError("RemoteException while launching purchase flow for sku " + sku);
            DebugLog.logException(e);
            flagEndAsync();

            deliverResult(listener, null, IABHELPER_REMOTE_EXCEPTION, "Remote exception while starting purchase flow");
        }
    }

    /**
     * Handles an activity result that's part of the purchase flow in in-app billing. If you
     * are calling {@link #launchPurchaseFlow}, then you must call this method from your
     * Activity's {@link android.app.Activity@onActivityResult} method. This method
     * MUST be called from the UI thread of the Activity.
     *
     * @param requestCode The requestCode as you received it.
     * @param resultCode  The resultCode as you received it.
     * @param data        The data (Intent) as you received it.
     * @return Returns true if the result was related to a purchase flow and was handled;
     * false if the result was not related to a purchase, in which case you should
     * handle it normally.
     */
    public boolean handleActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode != this.requestCode) {
            return false;
        }

        ensureNotDisposed();
        ensureSetupDone();

        // end of async purchase operation that started on launchPurchaseFlow
        flagEndAsync();

        if (data == null) {
            logError("Null data in IAB activity result.");
            deliverResult(purchaseListener, null, IABHELPER_BAD_RESPONSE, "Null data in IAB result");
            return true;
        }

        int responseCode = getResponseCodeFromIntent(data);
        String purchaseData = data.getStringExtra(RESPONSE_INAPP_PURCHASE_DATA);
        String dataSignature = data.getStringExtra(RESPONSE_INAPP_SIGNATURE);

        if (resultCode == Activity.RESULT_OK && responseCode == BILLING_RESPONSE_RESULT_OK) {
            logDebug("Successful resultcode from purchase activity.");
            logDebug("Purchase data: " + purchaseData);
            logDebug("Data signature: " + dataSignature);
            logDebug("Extras: " + data.getExtras());
            logDebug("Expected item type: " + purchasingItemType);

            if (purchaseData == null || dataSignature == null) {
                logError("BUG: either purchaseData or dataSignature is null.");
                logDebug("Extras: " + data.getExtras().toString());
                deliverResult(purchaseListener, null, IABHELPER_UNKNOWN_ERROR, "IAB returned null purchaseData or dataSignature");
                return true;
            }

            Purchase purchase = null;
            String sku = null;
            try {
                purchase = new Purchase(purchasingItemType, purchaseData, dataSignature);
                sku = purchase.getSku();

                String storedPayload = preferences.getString(PreferencesConstants.SKU_DEVELOPER_PAYLOAD_PREFIX + sku, "");
                if (!TextUtils.isEmpty(purchase.getDeveloperPayload())
                        && storedPayload.equals(purchase.getDeveloperPayload())) {
                    preferences.edit().remove(PreferencesConstants.SKU_DEVELOPER_PAYLOAD_PREFIX + sku).commit();
                } else {
                    logError("Purchase developer payload verification FAILED for sku " + sku);
                    deliverResult(purchaseListener, purchase, IABHELPER_VERIFICATION_FAILED, "Developer payload verification failed for sku " + sku);
                    return true;
                }

                // Verify signature
                securityManager.verifySignature(purchaseData, dataSignature);

                logDebug("Purchase signature successfully verified.");
            } catch (JSONException e) {
                logError("Failed to parse purchase data.");
                DebugLog.logException(e);
                deliverResult(purchaseListener, null, IABHELPER_BAD_RESPONSE, "Failed to parse purchase data.");
                return true;
            } catch (SecurityManager.VerificationFailedException e) {
                logError("Purchase signature verification FAILED for sku " + sku);
                DebugLog.logException(e);
                deliverResult(purchaseListener, null, IABHELPER_VERIFICATION_FAILED, "Signature verification failed for sku " + sku);
                return true;
            }

            deliverResult(purchaseListener, purchase, BILLING_RESPONSE_RESULT_OK, "Success");
        } else if (resultCode == Activity.RESULT_OK) {
            // result code was OK, but in-app billing response was not OK.
            logDebug("Result code was OK but in-app billing response was not OK: " + getResponseDesc(responseCode));
            deliverResult(purchaseListener, null, responseCode, "Problem purchashing item.");
        } else if (resultCode == Activity.RESULT_CANCELED) {
            logDebug("Purchase canceled - Response: " + getResponseDesc(responseCode));
            deliverResult(purchaseListener, null, IABHELPER_USER_CANCELLED, "User canceled.");
        } else {
            logError("Purchase failed. Result code: " + Integer.toString(resultCode)
                    + ". Response: " + getResponseDesc(responseCode));
            deliverResult(purchaseListener, null, IABHELPER_UNKNOWN_PURCHASE_RESPONSE, "Unknown purchase response.");
        }
        return true;
    }

    private void deliverResult(OnIabPurchaseFinishedListener listener, Purchase purchase, int resultCode, String resultMessage) {
        if (listener != null) {
            listener.onIabPurchaseFinished(new IabResult(resultCode, resultMessage), purchase);
        }
    }


    /**
     * Same as {@link #queryInventory(boolean, java.util.List, java.util.List)},
     * but without sub skus
     * @param querySkuDetails
     * @param moreSkus
     * @return {@link net.doo.snap.lib.util.billing.Inventory} instance
     * @throws IabException
     */
    public Inventory queryInventory(boolean querySkuDetails, List<String> moreSkus) throws IabException {
        return queryInventory(querySkuDetails, moreSkus, null);
    }

    /**
     * Queries the inventory. This will query all owned items from the server, as well as
     * information on additional skus, if specified. This method may block or take long to execute.
     * Do not call from a UI thread.
     *
     * @param querySkuDetails if true, SKU details (price, description, etc) will be queried as well
     *                        as purchase information.
     * @param moreItemSkus    additional PRODUCT skus to query information on, regardless of ownership.
     *                        Ignored if null or if querySkuDetails is false.
     * @param moreSubsSkus    additional SUBSCRIPTIONS skus to query information on, regardless of ownership.
     *                        Ignored if null or if querySkuDetails is false.
     * @throws IabException if a problem occurs while refreshing the inventory.
     */
    public Inventory queryInventory(boolean querySkuDetails, List<String> moreItemSkus,
                                    List<String> moreSubsSkus) throws IabException {
        ensureNotDisposed();
        ensureSetupDone();
        try {
            Inventory inv = new Inventory();
            int result = queryPurchases(inv, ITEM_TYPE_INAPP);
            if (result != BILLING_RESPONSE_RESULT_OK) {
                throw new IabException(result, "Error refreshing inventory (querying owned items).");
            }

            if (querySkuDetails) {
                result = querySkuDetails(ITEM_TYPE_INAPP, inv, moreItemSkus);
                if (result != BILLING_RESPONSE_RESULT_OK) {
                    throw new IabException(result, "Error refreshing inventory (querying prices of items).");
                }
            }

            // if subscriptions are supported, then also query for subscriptions
            if (subscriptionsSupported) {
                result = queryPurchases(inv, ITEM_TYPE_SUBS);
                if (result != BILLING_RESPONSE_RESULT_OK) {
                    throw new IabException(result, "Error refreshing inventory (querying owned subscriptions).");
                }

                if (querySkuDetails) {
                    result = querySkuDetails(ITEM_TYPE_SUBS, inv, moreItemSkus);
                    if (result != BILLING_RESPONSE_RESULT_OK) {
                        throw new IabException(result, "Error refreshing inventory (querying prices of subscriptions).");
                    }
                }
            }

            return inv;
        } catch (RemoteException e) {
            throw new IabException(IABHELPER_REMOTE_EXCEPTION, "Remote exception while refreshing inventory.", e);
        } catch (JSONException e) {
            throw new IabException(IABHELPER_BAD_RESPONSE, "Error parsing JSON response while refreshing inventory.", e);
        }
    }

    /**
     * Consumes a given in-app product. Consuming can only be done on an item
     * that's owned, and as a result of consumption, the user will no longer own it.
     * This method may block or take long to return. Do not call from the UI thread.
     *
     * @param itemInfo The PurchaseInfo that represents the item to consume.
     * @throws IabException if there is a problem during consumption.
     */
    public void consume(Purchase itemInfo) throws IabException {
        ensureNotDisposed();
        ensureSetupDone();

        if (!itemInfo.getItemType().equals(ITEM_TYPE_INAPP)) {
            throw new IabException(IABHELPER_INVALID_CONSUMPTION,
                    "Items of type '" + itemInfo.getItemType() + "' can't be consumed.");
        }

        try {
            String token = itemInfo.getToken();
            String sku = itemInfo.getSku();
            if (TextUtils.isEmpty(token)) {
                logError("Can't consume " + sku + ". No token.");
                throw new IabException(IABHELPER_MISSING_TOKEN, "PurchaseInfo is missing token for sku: "
                        + sku + " " + itemInfo);
            }

            logDebug("Consuming sku: " + sku + ", token: " + token);
            int response = service.consumePurchase(BILLING_API_VERSION, context.getPackageName(), token);
            if (response == BILLING_RESPONSE_RESULT_OK) {
                logDebug("Successfully consumed sku: " + sku);
            } else {
                logDebug("Error consuming consuming sku " + sku + ". " + getResponseDesc(response));
                throw new IabException(response, "Error consuming sku " + sku);
            }
        } catch (RemoteException e) {
            throw new IabException(IABHELPER_REMOTE_EXCEPTION, "Remote exception while consuming. PurchaseInfo: " + itemInfo, e);
        }
    }

    /**
     * Returns a human-readable description for the given response code.
     *
     * @param code The response code
     * @return A human-readable string explaining the result code.
     * It also includes the result code numerically.
     */
    public static String getResponseDesc(int code) {
        String[] iabResponses = {"0:OK",
                "1:User Canceled",
                "2:Unknown",
                "3:Billing Unavailable",
                "4:Item unavailable",
                "5:Developer Error",
                "6:Error",
                "7:Item Already Owned",
                "8:Item not owned"
        };
        String[] iabHelperResponses = {
                "0:OK",
                "-1001:Remote exception during initialization",
                "-1002:Bad response received",
                "-1003:Purchase signature verification failed",
                "-1004:Send intent failed",
                "-1005:User cancelled",
                "-1006:Unknown purchase response",
                "-1007:Missing token",
                "-1008:Unknown error",
                "-1009:Subscriptions not available",
                "-1010:Invalid consumption attempt"
        };

        if (code <= IABHELPER_ERROR_BASE) {
            int index = IABHELPER_ERROR_BASE - code;
            if (index >= 0 && index < iabHelperResponses.length) {
                return iabHelperResponses[index];
            } else {
                return String.valueOf(code) + ":Unknown IAB Helper Error";
            }
        } else if (code < 0 || code >= iabResponses.length) {
            return String.valueOf(code) + ":Unknown";
        } else {
            return iabResponses[code];
        }
    }


    // Checks that setup was done; if not, throws an exception.
    private void ensureSetupDone() {
        if (setupDone.compareAndSet(false, false)) {
            logError("Illegal state for operation: IAB helper is not set up.");
            throw new IllegalStateException("IAB helper is not set up. Can't perform operation.");
        }
    }

    // Workaround to bug where sometimes response codes come as Long instead of Integer
    private int getResponseCodeFromBundle(Bundle bundle) {
        Object object = bundle.get(RESPONSE_CODE);
        if (object == null) {
            logDebug("Bundle with null response code, assuming OK (known issue)");
            return BILLING_RESPONSE_RESULT_OK;
        } else if (object instanceof Integer) {
            return (Integer) object;
        } else if (object instanceof Long) {
            return (int) ((Long) object).longValue();
        } else {
            logError("Unexpected type for bundle response code.");
            logError(object.getClass().getName());
            throw new RuntimeException("Unexpected type for bundle response code: " + object.getClass().getName());
        }
    }

    // Workaround to bug where sometimes response codes come as Long instead of Integer
    private int getResponseCodeFromIntent(Intent intent) {
        return getResponseCodeFromBundle(intent.getExtras());
    }

    private void flagStartAsync() {
        if (asyncInProgress.getAndSet(true)) {
            throw new IllegalStateException("Can't start async operation because another async operation is in progress.");
        }
        logDebug("Starting async operation");
    }

    private void flagEndAsync() {
        logDebug("Ending async operation");
        asyncInProgress.set(false);
    }


    private int queryPurchases(Inventory inv, String itemType) throws JSONException, RemoteException {
        // Query purchases
        logDebug("Querying owned items, item type: " + itemType);
        logDebug("Package name: " + context.getPackageName());
        boolean verificationFailed = false;
        String continueToken = null;

        do {
            logDebug("Calling getPurchases with continuation token: " + continueToken);
            Bundle ownedItems = service.getPurchases(BILLING_API_VERSION, context.getPackageName(),
                    itemType, continueToken);

            int response = getResponseCodeFromBundle(ownedItems);
            logDebug("Owned items response: " + String.valueOf(response));
            if (response != BILLING_RESPONSE_RESULT_OK) {
                logDebug("getPurchases() failed: " + getResponseDesc(response));
                return response;
            }
            if (!ownedItems.containsKey(RESPONSE_INAPP_ITEM_LIST)
                    || !ownedItems.containsKey(RESPONSE_INAPP_PURCHASE_DATA_LIST)
                    || !ownedItems.containsKey(RESPONSE_INAPP_SIGNATURE_LIST)) {
                logError("Bundle returned from getPurchases() doesn't contain required fields.");
                return IABHELPER_BAD_RESPONSE;
            }

            ArrayList<String> ownedSkus = ownedItems.getStringArrayList(
                    RESPONSE_INAPP_ITEM_LIST);
            ArrayList<String> purchaseDataList = ownedItems.getStringArrayList(
                    RESPONSE_INAPP_PURCHASE_DATA_LIST);
            ArrayList<String> signatureList = ownedItems.getStringArrayList(
                    RESPONSE_INAPP_SIGNATURE_LIST);

            for (int i = 0; i < purchaseDataList.size(); ++i) {
                String purchaseData = purchaseDataList.get(i);
                String signature = signatureList.get(i);
                String sku = ownedSkus.get(i);
                try {
                    securityManager.verifySignature(purchaseData, signature);
                    logDebug("Sku is owned: " + sku);
                    Purchase purchase = new Purchase(itemType, purchaseData, signature);

                    if (TextUtils.isEmpty(purchase.getToken())) {
                        logWarn("BUG: empty/null token!");
                        logDebug("Purchase data: " + purchaseData);
                    }

                    // Record ownership and token
                    inv.addPurchase(purchase);
                } catch (SecurityManager.VerificationFailedException e) {
                    DebugLog.logException(e);
                    logWarn("Purchase signature verification **FAILED**. Not adding item.");
                    logDebug("   Purchase data: " + purchaseData);
                    logDebug("   Signature: " + signature);
                    verificationFailed = true;
                }
            }

            continueToken = ownedItems.getString(INAPP_CONTINUATION_TOKEN);
            logDebug("Continuation token: " + continueToken);
        } while (!TextUtils.isEmpty(continueToken));

        return verificationFailed ? IABHELPER_VERIFICATION_FAILED : BILLING_RESPONSE_RESULT_OK;
    }

    private int querySkuDetails(String itemType, Inventory inv, List<String> moreSkus)
            throws RemoteException, JSONException {
        logDebug("Querying SKU details.");
        ArrayList<String> skuList = new ArrayList<String>();
        skuList.addAll(inv.getAllOwnedSkus(itemType));
        if (moreSkus != null) {
            for (String sku : moreSkus) {
                if (!skuList.contains(sku)) {
                    skuList.add(sku);
                }
            }
        }

        if (skuList.size() == 0) {
            logDebug("queryPrices: nothing to do because there are no SKUs.");
            return BILLING_RESPONSE_RESULT_OK;
        }

        Bundle querySkus = new Bundle();
        querySkus.putStringArrayList(GET_SKU_DETAILS_ITEM_LIST, skuList);
        Bundle skuDetails = service.getSkuDetails(BILLING_API_VERSION, context.getPackageName(),
                itemType, querySkus);

        if (!skuDetails.containsKey(RESPONSE_GET_SKU_DETAILS_LIST)) {
            int response = getResponseCodeFromBundle(skuDetails);
            if (response != BILLING_RESPONSE_RESULT_OK) {
                logDebug("getSkuDetails() failed: " + getResponseDesc(response));
                return response;
            } else {
                logError("getSkuDetails() returned a bundle with neither an error nor a detail list.");
                return IABHELPER_BAD_RESPONSE;
            }
        }

        ArrayList<String> responseList = skuDetails.getStringArrayList(
                RESPONSE_GET_SKU_DETAILS_LIST);

        for (String thisResponse : responseList) {
            SkuDetails d = new SkuDetails(itemType, thisResponse);
            logDebug("Got sku details: " + d);
            inv.addSkuDetails(d);
        }
        return BILLING_RESPONSE_RESULT_OK;
    }

    private void logDebug(String msg) {
        DebugLog.d(msg);
    }

    private void logError(String msg) {
        DebugLog.e("In-app billing error: " + msg);
    }

    private void logWarn(String msg) {
        DebugLog.w("In-app billing warning: " + msg);
    }

}
