package net.doo.snap.lib.snap.preview;

import android.animation.AnimatorSet;
import android.animation.ObjectAnimator;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Paint;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.RectShape;
import android.os.AsyncTask;
import android.os.Bundle;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.Loader;
import android.util.DisplayMetrics;
import android.view.GestureDetector;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.ImageView;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.detector.ContourDetector;
import net.doo.snap.lib.edit.EditPolygonActivity;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageStoreStrategy;
import net.doo.snap.lib.persistence.PictureProcessor;
import net.doo.snap.lib.persistence.RotationType;
import net.doo.snap.lib.snap.edit.EditLock;
import net.doo.snap.lib.snap.edit.drag.PagesRearranger;
import net.doo.snap.lib.snap.edit.events.DeletePageEvent;
import net.doo.snap.lib.snap.event.PageOptimizationTypeChangedEvent;
import net.doo.snap.lib.snap.preview.zoom.ZoomingManager;
import net.doo.snap.lib.ui.SnappingFragment;
import net.doo.snap.lib.ui.events.OnPageLongPressedEvent;
import net.doo.snap.lib.ui.util.SwipeDownTouchListener;
import net.doo.snap.lib.ui.util.TransformableDrawable;
import net.doo.snap.lib.ui.util.ViewUtils;
import net.doo.snap.lib.util.State;
import net.doo.snap.lib.util.StateHolder;
import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.bitmap.PageBitmapLoader;
import net.doo.snap.lib.util.log.DebugLog;
import net.doo.snap.lib.util.ui.CompositeTouchListener;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.lang.ref.WeakReference;

import roboguice.activity.event.OnActivityResultEvent;
import roboguice.event.EventManager;
import roboguice.event.Observes;
import roboguice.fragment.RoboFragment;

/**
 * Shows preview of snapped image
 *
 * @see net.doo.snap.lib.snap.SnappingActivity
 */
public class ImagePreviewFragment extends RoboFragment implements LoaderManager.LoaderCallbacks<Bitmap> {

    private static final long DROPDOWN_ANIMATION_DURATION_MILLIS = 300;
    private static final String PAGE_DATA = "PAGE_DATA";
    private static final float DROP_TARGET_INITIAL_SCALE_X = 0.8f;
    private static final float DROP_TARGET_INITIAL_SCALE_Y = 0.5f;
    private static final String PAGE_PATH = "PAGE_PATH";

    /**
     * {@code true} to play "drop in" animation on fragment appearance, {@code false} to disable animation
     */
    public static final String PLAY_DROP_IN_ANIMATION = "PLAY_DROP_IN_ANIMATION";
    /**
     * initial X coordinate of fragment's center for "drop in" animation
     */
    public static final String DROP_X = "DROP_X";
    /**
     * initial Y coordinate of fragment's center for "drop in" animation
     */
    public static final String DROP_Y = "DROP_Y";
    /**
     * {@link android.graphics.Bitmap} which might be used to display during drop-in animation. Note, that
     * {@link ImagePreviewFragment} will recycle it manually, without relying on reference counter.
     */
    public static final String PRESERVED_BITMAP = "PRESERVED_BITMAP";

    private ImageView thumbnail;
    private Bitmap bitmap;

    private View leftDropTarget;
    private View rightDropTarget;
    private View progressView;

    @Inject
    private EventManager eventManager;
    @Inject
    private PageStoreStrategy pageStoreStrategy;
    @Inject
    private BitmapLruCache bitmapLruCache;
    @Inject
    private EditLock editLock;
    @Inject
    private ZoomingManager zoomingManager;

    private float dropdownTargetWidth;

    private Page page;

    private boolean useDropInAnimation;
    private float dropX;
    private float dropY;

    private boolean selected = false;

    private State SHOW_LEFT_DROP_TARGET = new State() {

        @Override
        public void onEnter() {
            offsetThumbnail(dropdownTargetWidth);
            showDropTargetWithAnimation(leftDropTarget);
        }

        @Override
        public void onLeave() {
            offsetThumbnail(ViewUtils.TRANSLATION_DEFAULT);
            leftDropTarget.animate().alpha(ViewUtils.ALPHA_TRANSPARENT).start();
        }
    };

    private State SHOW_RIGHT_DROP_TARGET = new State() {

        @Override
        public void onEnter() {
            offsetThumbnail(-dropdownTargetWidth);
            showDropTargetWithAnimation(rightDropTarget);
        }

        @Override
        public void onLeave() {
            offsetThumbnail(ViewUtils.TRANSLATION_DEFAULT);
            rightDropTarget.animate().alpha(ViewUtils.ALPHA_TRANSPARENT).start();
        }

    };

    private StateHolder stateHolder = new StateHolder();

    @Nullable
    private TransformableDrawable transformableDrawable;


    /**
     * @param page information, required for this fragment to display data
     * @return new instance of {@link ImagePreviewFragment}
     */
    public static ImagePreviewFragment newInstance(Page page) {
        Bundle pageParams = page.getParameters();
        ImagePreviewFragment fragment = new ImagePreviewFragment(
                pageParams.getBoolean(PLAY_DROP_IN_ANIMATION),
                pageParams.getFloat(DROP_X),
                pageParams.getFloat(DROP_Y),
                (Bitmap) pageParams.getParcelable(PRESERVED_BITMAP)
        );
        pageParams.remove(PRESERVED_BITMAP);
        pageParams.remove(DROP_X);
        pageParams.remove(DROP_Y);
        pageParams.remove(PLAY_DROP_IN_ANIMATION);

        Bundle args = new Bundle();
        args.putParcelable(PAGE_DATA, page);
        fragment.setArguments(args);

        return fragment;
    }

    /**
     * Used by system
     */
    public ImagePreviewFragment() {
    }

    /**
     * Used to pass animation parameters, which shouldn't be saved between configuration changes
     */
    @SuppressLint("ValidFragment")
    public ImagePreviewFragment(boolean useDropInAnimation, float dropX, float dropY, Bitmap preservedBitmap) {
        this.useDropInAnimation = useDropInAnimation;
        this.dropX = dropX;
        this.dropY = dropY;
        this.bitmap = preservedBitmap;
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        Bundle args = getArguments();
        page = args.getParcelable(PAGE_DATA);

        Resources resources = activity.getResources();
        dropdownTargetWidth = resources.getDimension(R.dimen.page_dropdown_target_width);
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        try {
            Bundle args = new Bundle();
            final String optimizedPreviewPath = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.OPTIMIZED_PREVIEW).getPath();
            args.putString(PAGE_PATH, optimizedPreviewPath);
            getLoaderManager().initLoader(0, args, this);
        } catch (IOException e) {
            DebugLog.logException(e);
        }
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.image_preview_fragment, container, false);
        restoreState(savedInstanceState);

        progressView = view.findViewById(R.id.progress_view);

        thumbnail = (ImageView) view.findViewById(R.id.thumbnail);
        if (bitmap == null) {
            thumbnail.setImageDrawable(getPagePlaceholder());
            progressView.setVisibility(View.VISIBLE);
        } else {
            thumbnail.setImageBitmap(bitmap);
            progressView.setVisibility(View.GONE);
        }

        leftDropTarget = view.findViewById(R.id.drop_target_left);
        rightDropTarget = view.findViewById(R.id.drop_target_right);

        zoomingManager.init(thumbnail, new ZoomingManager.ZoomingCallbacks() {

            @NotNull
            @Override
            public Drawable getZoomedDrawable() {
                if (transformableDrawable == null) {
                    throw new NullPointerException("Drawable is null");
                }

                TransformableDrawable drawable = new TransformableDrawable(
                        new BitmapDrawable(getResources(), bitmap)
                );
                drawable.setRotation(transformableDrawable.getRotation());
                drawable.setScale(transformableDrawable.getScale());
                drawable.setAdjustBounds(true);

                return drawable;
            }

            @Override
            public Object getToken() {
                return page;
            }

            @Override
            public boolean isZoomingEnabled() {
                return selected && bitmap != null && transformableDrawable != null;
            }

        });

        return view;
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putParcelable(PAGE_DATA, page);
    }

    private void restoreState(Bundle savedInstanceState) {
        if (savedInstanceState == null) {
            return;
        }

        page = savedInstanceState.getParcelable(PAGE_DATA);
    }

    @Override
    public void onViewCreated(View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        SwipeDownTouchListener swipeDownTouchListener = new SwipeDownTouchListener(view) {
            @Override
            protected boolean onSwipeDownDetected() {
                if (!editLock.isEditLocked()) {
                    eventManager.fire(new DeletePageEvent());
                    return true;
                } else {
                    return false;
                }
            }

            @Override
            public boolean onTouch(View v, MotionEvent event) {
                return !editLock.isEditLocked() && super.onTouch(v, event);
            }
        };

        view.setOnTouchListener(
                new CompositeTouchListener() {
                    @Override
                    public boolean onTouch(View view, MotionEvent event) {
                        return selected && super.onTouch(view, event);
                    }
                }
                        .addGestureDetector(new GestureDetector(getActivity(), new GestureListener()))
                        .addScaleGestureDetector(zoomingManager.buildScaleDetector())
                        .addGestureDetector(zoomingManager.buildDobuleTapDetector())
                        .addListener(swipeDownTouchListener)
        );

        leftDropTarget.setAlpha(ViewUtils.ALPHA_TRANSPARENT);
        rightDropTarget.setAlpha(ViewUtils.ALPHA_TRANSPARENT);

        if (useDropInAnimation) {
            thumbnail.getViewTreeObserver().addOnPreDrawListener(new ViewTreeObserver.OnPreDrawListener() {
                @Override
                public boolean onPreDraw() {
                    try {
                        playDropInAnimation();
                        return true;
                    } finally {
                        thumbnail.getViewTreeObserver().removeOnPreDrawListener(this);
                    }
                }
            });
        }
    }

    private void playDropInAnimation() {
        DisplayMetrics displayMetrics = getResources().getDisplayMetrics();

        final View rootView = getView();
        rootView.setTranslationX(dropX - displayMetrics.widthPixels / 2f);
        rootView.setTranslationY(dropY - rootView.getHeight() / 2f);
        rootView.setScaleX(PagesRearranger.DRAG_SHADOW_SCALE);
        rootView.setScaleY(PagesRearranger.DRAG_SHADOW_SCALE);
        rootView.setAlpha(PagesRearranger.DRAG_SHADOW_ALPHA);

        // Animating back to normal state
        rootView.animate()
                .scaleX(ViewUtils.SCALE_DEFAULT)
                .scaleY(ViewUtils.SCALE_DEFAULT)
                .translationX(ViewUtils.TRANSLATION_DEFAULT)
                .translationY(ViewUtils.TRANSLATION_DEFAULT)
                .alpha(ViewUtils.ALPHA_DEFAULT)
                .start();

        useDropInAnimation = false;
    }

    /**
     * Moves thumbnail to the right, and shows left drop zone with animation
     */
    public void showLeftDropTarget() {
        stateHolder.updateState(SHOW_LEFT_DROP_TARGET);
    }

    /**
     * Moves thumbnail to the left, and shows right drop zone with animation
     */
    public void showRightDropTarget() {
        stateHolder.updateState(SHOW_RIGHT_DROP_TARGET);
    }

    /**
     * Moves thumbnail to center (default state) and hides all drop zones
     */
    public void hideDropTarget() {
        stateHolder.updateState(State.DEFAULT);
    }

    private void offsetThumbnail(float offset) {
        thumbnail.animate()
                .translationX(offset)
                .setDuration(DROPDOWN_ANIMATION_DURATION_MILLIS)
                .setListener(null)
                .start();
    }

    private void showDropTargetWithAnimation(View dropTarget) {
        dropTarget.setScaleX(DROP_TARGET_INITIAL_SCALE_X);
        dropTarget.setScaleY(DROP_TARGET_INITIAL_SCALE_Y);

        dropTarget.animate()
                .alpha(ViewUtils.ALPHA_DEFAULT)
                .scaleX(ViewUtils.SCALE_DEFAULT)
                .scaleY(ViewUtils.SCALE_DEFAULT)
                .start();
    }

    /**
     * @param selected {@code true} if {@link ImagePreviewFragment} currently selected in view pager.
     *                 {@code false} otherwise
     */
    public void setSelected(boolean selected) {
        this.selected = selected;
    }

    /**
     * Changes {@link net.doo.snap.lib.persistence.RotationType} of associated {@link net.doo.snap.lib.persistence.Page}
     * and starts rotation animation.
     */
    public void setPageRotation(RotationType rotationType) {
        final Drawable drawable = thumbnail.getDrawable();
        if (drawable instanceof TransformableDrawable) {

            final float scale = ViewUtils.fitBitmapToView(bitmap, thumbnail, rotationType.getDegrees());

            AnimatorSet animatorSet = new AnimatorSet();
            animatorSet.playTogether(
                    ObjectAnimator.ofFloat(drawable, TransformableDrawable.PROPERTY_SCALE, scale),
                    ObjectAnimator.ofFloat(drawable, TransformableDrawable.PROPERTY_ROTATION, calculateRotationAngle(rotationType))
            );
            animatorSet.start();
        }

        page.setRotationType(rotationType);
    }

    private float calculateRotationAngle(RotationType rotationType) {
        if (page.getRotationType() == RotationType.ROTATION_360
                && rotationType == RotationType.ROTATION_90
                && thumbnail.getDrawable() instanceof TransformableDrawable) {
            ((TransformableDrawable) thumbnail.getDrawable()).setRotation(0f);
        }

        return rotationType.getDegrees();
    }

    /**
     * @return information to be displayed in this fragment
     */
    public Page getPage() {
        return page;
    }


    private void initThumbnailParams() {
        Drawable drawable = thumbnail.getDrawable();
        if (!(drawable instanceof TransformableDrawable)) {
            return;
        }

        final RotationType rotationType = page.getRotationType();
        final float scale = ViewUtils.fitBitmapToView(bitmap, thumbnail, rotationType.getDegrees());

        TransformableDrawable transformableDrawable = (TransformableDrawable) drawable;
        transformableDrawable.setScale(scale);
        transformableDrawable.setRotation(calculateRotationAngle(page.getRotationType()));
        transformableDrawable.setAdjustBounds(true);
    }

    @Override
    public Loader<Bitmap> onCreateLoader(int id, Bundle args) {
        String path = args.getString(PAGE_PATH);
        return new PageBitmapLoader(
                getActivity(),
                path,
                bitmapLruCache,
                bitmap
        );
    }

    @Override
    public void onLoadFinished(Loader<Bitmap> loader, Bitmap data) {
        if (!isAdded()) {
            return;
        }

        bitmap = data;
        onDataReady();
    }

    private void onDataReady() {
        transformableDrawable = new TransformableDrawable(
                new BitmapDrawable(getResources(), bitmap)
        );
        thumbnail.setImageDrawable(transformableDrawable);
        ViewUtils.postOnPreDraw(thumbnail, new Runnable() {
            @Override
            public void run() {
                initThumbnailParams();
            }
        });
        progressView.setVisibility(View.GONE);
    }

    @Override
    public void onLoaderReset(Loader<Bitmap> loader) {
        bitmap = null;
        transformableDrawable = null;

        thumbnail.setImageDrawable(getPagePlaceholder());
        progressView.setVisibility(View.VISIBLE);
    }

    private Drawable getPagePlaceholder() {
        ShapeDrawable drawable = new ShapeDrawable(new RectShape());

        /*
        This lines fixes bug for 4.0.3 (ICS) devices. Initial padding Rect is null,
        which causes app to crash.
         */
        drawable.setPadding(new Rect());

        Paint paint = drawable.getPaint();
        paint.setColor(getResources().getColor(R.color.page_placeholder));
        paint.setStyle(Paint.Style.FILL);

        Point imageSize = page.getImageSize(Page.ImageType.OPTIMIZED_PREVIEW);
        drawable.setIntrinsicWidth(imageSize.x);
        drawable.setIntrinsicHeight(imageSize.y);

        return drawable;
    }

    /**
     * This event is called when activity result is returned,
     * <p/>
     * Real event is fired from {@link ImagePreviewFragment}
     * because system event returns wrong request code
     */
    @SuppressWarnings("unused")
    private void editPolygonFinished(@Observes OnActivityResultEvent event) {
        if (event.getResultCode() == Activity.RESULT_OK
                && event.getRequestCode() == SnappingFragment.EDIT_ACTIVITY_REQUEST_CODE) {
            Page resultPage = event.getData().getParcelableExtra(EditPolygonActivity.PAGE);
            if (page.equals(resultPage)) {
                restartLoader();
            }
        }
    }

    @SuppressWarnings("unused")
    public void onOptimizationTypeChanged(@Observes PageOptimizationTypeChangedEvent event) {
        if (!page.equals(event.getPage())) {
            return;
        }

        try {
            bitmapLruCache.remove(pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.OPTIMIZED_PREVIEW).getPath());
        } catch (IOException e) {
            DebugLog.logException(e);
        }

        page.setOptimizationType(event.getOptimizationType());
        new GenerateOptimizedPreviewTask(page, editLock, pageStoreStrategy, progressView).execute();
    }

    private void restartLoader() {
        try {
            final String optimizedPreviewPath = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.OPTIMIZED_PREVIEW).getPath();

            Bundle args = new Bundle();
            args.putString(PAGE_PATH, optimizedPreviewPath);
            bitmapLruCache.remove(optimizedPreviewPath);
            bitmap = null;
            getLoaderManager().restartLoader(0, args, this);
        } catch (IOException e) {
            DebugLog.logException(e);
        }
    }

    @Override
    public void onDestroy() {
        /*
        Temporary workaround. EventManager holds strong references to fragments.
        See https://github.com/roboguice/roboguice/pull/184
         */
        eventManager.unregisterObserver(this, PageOptimizationTypeChangedEvent.class);
        eventManager.unregisterObserver(this, OnActivityResultEvent.class);

        super.onDestroy();
    }

    /**
     * Detects following gestures:
     * - Long press, to start dragging
     * - Double-tap to zoom page in
     */
    private class GestureListener extends GestureDetector.SimpleOnGestureListener {

        @Override
        public void onLongPress(MotionEvent motionEvent) {
            if (!stateHolder.getCurrentState().equals(State.DEFAULT)) {
                return;
            }

            eventManager.fire(new OnPageLongPressedEvent(
                    motionEvent.getX() - thumbnail.getLeft(),
                    motionEvent.getY() - thumbnail.getTop(),
                    thumbnail,
                    page,
                    ImagePreviewFragment.this,
                    bitmap
            ));
        }
    }

    /**
     * Generates new bitmap preview for {@link net.doo.snap.lib.persistence.Page}
     */
    private static class GenerateOptimizedPreviewTask extends AsyncTask<Void, Void, Void> {

        private Page page;
        private EditLock editLock;
        private PageStoreStrategy pageStoreStrategy;
        private WeakReference<View> progressViewReference;

        private GenerateOptimizedPreviewTask(Page page, EditLock editLock, PageStoreStrategy pageStoreStrategy, View progressView) {
            this.page = page;
            this.progressViewReference = new WeakReference<>(progressView);
            this.editLock = editLock;
            this.pageStoreStrategy = pageStoreStrategy;
        }

        @Override
        protected void onPreExecute() {
            editLock.lockEdit();

            View progressView = progressViewReference.get();
            if (progressView != null) {
                progressView.setVisibility(View.VISIBLE);
            }
        }

        @Override
        protected Void doInBackground(Void... voids) {
            try {
                PictureProcessor.generateOptimizedPreview(
                        pageStoreStrategy,
                        page,
                        new ContourDetector()
                );
            } catch (IOException e) {
                DebugLog.logException(e);
            }

            return null;
        }

        @Override
        protected void onCancelled() {
            super.onCancelled();

            editLock.unlockEdit();
        }

        @Override
        protected void onPostExecute(Void result) {
            editLock.unlockEdit();

            View progressView = progressViewReference.get();
            if (progressView != null) {
                progressView.setVisibility(View.GONE);
            }
        }
    }

}
