package net.doo.snap.lib.snap.preview.zoom;

import android.app.Activity;
import android.graphics.drawable.Drawable;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;

import com.google.inject.Inject;

import net.doo.snap.lib.snap.edit.EditLock;
import net.doo.snap.lib.ui.util.ViewUtils;

import org.jetbrains.annotations.NotNull;

import roboguice.event.EventManager;
import roboguice.event.Observes;

/**
 * Helper class that controls zooming workflow.
 * Provides set-up {@link android.view.ScaleGestureDetector} and {@link android.view.GestureDetector} for scale gestures.
 */
public class ZoomingManager {

    /**
     * Provides information required to start zooming workflow
     */
    public interface ZoomingCallbacks {

        /**
         * @return {@link android.graphics.drawable.Drawable} which will be used for zooming
         */
        @NotNull
        Drawable getZoomedDrawable();

        /**
         * @return token that uniquely identifies entity being zoomed
         */
        Object getToken();

        /**
         * @return {@code true} if zooming is allowed. {@code false} otherwise.
         */
        boolean isZoomingEnabled();

    }

    @Inject
    private EventManager eventManager;
    @Inject
    private Activity activity;
    @Inject
    private EditLock editLock;

    private View initialView;
    private ZoomingCallbacks zoomingCallbacks;

    @Inject
    public ZoomingManager() {
    }

    /**
     * Initializes {@link net.doo.snap.lib.snap.preview.zoom.ZoomingManager}
     *
     * @param initialView      {@link android.view.View} that will be used for calculation of scaled preview position
     * @param zoomingCallbacks {@link net.doo.snap.lib.snap.preview.zoom.ZoomingManager.ZoomingCallbacks} which will provide information required for zooming workflow.
     */
    public void init(@NotNull View initialView, @NotNull ZoomingCallbacks zoomingCallbacks) {
        this.initialView = initialView;
        this.zoomingCallbacks = zoomingCallbacks;
    }

    /**
     * @return {@link android.view.ScaleGestureDetector} that will handle scaling events. You should assign it to initial view.
     */
    public ScaleGestureDetector buildScaleDetector() {
        ensureInitialized();
        return new ScaleGestureDetector(activity, new ScaleListener());
    }

    /**
     * @return {@link android.view.GestureDetector} that will handle double-tap events. You should assign it to initial view.
     */
    public GestureDetector buildDobuleTapDetector() {
        ensureInitialized();
        return new GestureDetector(activity, new GestureListener());
    }

    private boolean prepareToScaling() {
        if (!zoomingCallbacks.isZoomingEnabled()) {
            return false;
        }

        Drawable scaledDrawable = zoomingCallbacks.getZoomedDrawable();

        editLock.lockEdit();
        activity.getActionBar().hide();

        eventManager.fire(new PrepareZoomEvent(scaledDrawable, ViewUtils.getViewBounds(initialView), zoomingCallbacks.getToken()));
        initialView.setVisibility(View.INVISIBLE);

        return true;
    }

    private void startScaling() {
        eventManager.fire(new StartZoomingEvent());
    }

    private void quickScale() {
        eventManager.fire(new QuickScaleEvent());
    }

    @SuppressWarnings("unused")
    private void onScaleEnded(@Observes ZoomingFinishedEvent event) {
        if (zoomingCallbacks != null && event.getToken().equals(zoomingCallbacks.getToken())) {
            initialView.setVisibility(View.VISIBLE);
            activity.getActionBar().show();
            editLock.unlockEdit();
        }
    }

    private void ensureInitialized() {
        if (initialView == null || zoomingCallbacks == null) {
            throw new IllegalStateException("ZoomingManager is not initialized.");
        }
    }

    /**
     * Detects double-tap to zoom image in
     */
    private class GestureListener extends GestureDetector.SimpleOnGestureListener {

        @Override
        public boolean onSingleTapUp(MotionEvent e) {
            return true;
        }

        @Override
        public boolean onDoubleTap(MotionEvent e) {
            ensureInitialized();

            if (editLock.isEditLocked()) {
                return false;
            }

            if (!zoomingCallbacks.isZoomingEnabled()) {
                return false;
            }

            boolean prepared = prepareToScaling();
            if (!prepared) {
                return false;
            }

            quickScale();

            return true;
        }
    }

    /**
     * Controls scale gesture
     */
    private class ScaleListener extends ScaleGestureDetector.SimpleOnScaleGestureListener {

        private float currentScale;
        private PreScaleChangedEvent preScaleChangedEvent = new PreScaleChangedEvent();

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            ensureInitialized();

            if (editLock.isEditLocked()) {
                return false;
            }

            currentScale = ViewUtils.SCALE_DEFAULT;
            return prepareToScaling();

        }

        @Override
        public boolean onScale(ScaleGestureDetector detector) {
            currentScale *= detector.getScaleFactor();

            if (currentScale < ViewUtils.SCALE_DEFAULT) {
                currentScale = ViewUtils.SCALE_DEFAULT;
            }

            preScaleChangedEvent.scale = currentScale;
            eventManager.fire(preScaleChangedEvent);

            return true;
        }

        @Override
        public void onScaleEnd(ScaleGestureDetector detector) {
            startScaling();
        }
    }

}
