package net.doo.snap.lib.util.snap;

import android.graphics.Matrix;
import android.graphics.PointF;

import java.util.List;

/**
 * Helper to make it easier to draw polygon on view
 */
public class PolygonHelper {
    /**
     * points to rotate polygon depending on device orientation
     * view instance holds it to decrease garbage collector calls
     */
    private float[] tmpPoints = new float[8];

    /**
     * Rotation matrix
     */
    private Matrix matrix = new Matrix();

    private int width;
    private int height;

    private int left;
    private int top;
    private int right;
    private int bottom;

    /**
     * Set rotation in degrees for polygon
     * @param degrees usually camera rotation angle is used here
     */
    public void setRotation(int degrees) {
        matrix.setRotate(degrees, 0.5f, 0.5f);
    }

    /**
     * Set scaled image size that will be drawn on screen
     * @param width
     * @param height
     */
    public void setImageSize(int width, int height) {
        this.width = width;
        this.height = height;
    }

    /**
     * Sets layout of surface inside parent to compensate differences
     * between viewport and preview sizes and draw polygon correctly
     *
     * @param left
     * @param top
     * @param right
     * @param bottom
     */
    public void setLayout(int left, int top, int right, int bottom) {
        this.left = left;
        this.top = top;
        this.right = right;
        this.bottom = bottom;

        width = right - left;
        height = bottom - top;
    }

    /**
     * Calculates lines points from polygon multiplying 0..1 values by view dimensions
     * and applying rotation matrix.
     * Usually used on preview as it has camera rotation
     * @param polygon
     * @param points
     */
    public void polygonToPoints(List<PointF> polygon, float[] points) {
        for (int i = 0; i < polygon.size(); i++) {
            tmpPoints[i * 2] = polygon.get(i).x;
            tmpPoints[i * 2 + 1] = polygon.get(i).y;
        }

        matrix.mapPoints(tmpPoints);

        for (int i = 0; i < polygon.size(); i++) {
            points[i * 4] = left + width * tmpPoints[i * 2];
            points[i * 4 + 1] = top + height * tmpPoints[i * 2 + 1];
            points[i * 4 + 2] = left + width * tmpPoints[(i * 2 + 2) % 8];
            points[i * 4 + 3] = top + height * tmpPoints[(i * 2 + 3) % 8];
        }
    }

    /**
     * Processes original polygon with coordinates 0..1
     * to drawing polygon by multiplying them by view dimensions
     * @param polygon
     * @param drawingPolygon
     */
    public void getDrawingPolygon(List<PointF> polygon, List<PointF> drawingPolygon) {
        for (int i = 0; i < drawingPolygon.size(); i++) {
            PointF point = polygon.get(i);
            PointF drawingPoint = drawingPolygon.get(i);

            drawingPoint.x = point.x * width;
            drawingPoint.y = point.y * height;
        }
    }

    /**
     * Processes drawing polygon coordinates
     * to polygon with coordinates 0..1 by dividing them by view dimensions
     *
     * @param drawingPolygon
     * @param polygon
     */
    public void getPolygonFromDrawingPolygon(List<PointF> drawingPolygon, List<PointF> polygon) {
        for (int i = 0; i < drawingPolygon.size(); i++) {
            PointF point = polygon.get(i);
            PointF drawingPoint = drawingPolygon.get(i);

            point.x = drawingPoint.x / width;
            point.y = drawingPoint.y / height;
        }
    }
}
