package net.doo.snap.lib.ui;

import android.app.ActionBar;
import android.content.ClipboardManager;
import android.content.Context;
import android.content.Intent;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.support.v4.app.FragmentActivity;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.maps.CameraUpdateFactory;
import com.google.android.gms.maps.GoogleMap;
import com.google.android.gms.maps.MapView;
import com.google.android.gms.maps.MapsInitializer;
import com.google.android.gms.maps.model.LatLng;
import com.google.android.gms.maps.model.MarkerOptions;
import com.google.inject.Inject;
import com.google.zxing.Result;
import com.google.zxing.client.result.GeoParsedResult;
import com.google.zxing.client.result.ParsedResult;
import com.google.zxing.client.result.ParsedResultType;

import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.AnalyticsConst;
import net.doo.snap.lib.analytics.EasyTrackerWrapper;
import net.doo.snap.lib.snap.camera.barcode.BarcodeDetector;
import net.doo.snap.lib.snap.camera.barcode.result.ResultHandler;
import net.doo.snap.lib.snap.camera.barcode.result.ResultHandlerFactory;
import net.doo.snap.lib.ui.widget.text.CustomTypefaceTextView;

import roboguice.fragment.RoboFragment;

/**
 * Represents barcode content.
 */
public class BarcodeFragment extends RoboFragment {
    public static String BARCODE_FRAGMENT_TAG = "BARCODE_FRAGMENT";
    private static final String ACTION_BAR_TITLE = "ACTION_BAR_TITLE";
    private static final int DEFAULT_MAP_ZOOM = 15;

    @Inject
    private BarcodeDetector barcodeDetector;
    @Inject
    private Context context;

    private ImageView codeTypeImage;
    private Result lastCodeResult;
    private TextView title;
    private TextView subTitle;
    private TextView contentText;
    private LinearLayout buttonsContainer;
    private ResultHandler barcodeHandler;
    private WebView webContainer;
    private View contentContainer;
    private GoogleMap map;
    private MapView mapView;
    private Drawable actionBarBg;
    private String externalTitle;
    private ParsedResultType resultType;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setHasOptionsMenu(true);
        lastCodeResult = barcodeDetector.getLastResult();
        if (lastCodeResult == null) {
            getActivity().getSupportFragmentManager().popBackStack();
            return;
        }

        barcodeHandler = ResultHandlerFactory.makeResultHandler(getActivity(), lastCodeResult);
        ParsedResult result = barcodeHandler.getResult();
        if (result == null || result.getType() == null) {
            getActivity().getSupportFragmentManager().popBackStack();
            return;
        }
        resultType = result.getType();


        if (savedInstanceState != null) {
            externalTitle = savedInstanceState.getString(ACTION_BAR_TITLE, "");
        } else {
            ActionBar actionBar = getActivity().getActionBar();
            if (!TextUtils.isEmpty(actionBar.getTitle())) {
                externalTitle = actionBar.getTitle().toString();
            }
        }
        actionBarBg = getResources().getDrawable(R.color.action_bar_bg);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View rootView = inflater.inflate(R.layout.barcode_fragment, container, false);

        codeTypeImage = (ImageView) rootView.findViewById(R.id.type_icon);
        title = (TextView) rootView.findViewById(R.id.title);
        subTitle = (TextView) rootView.findViewById(R.id.subtitle);
        contentText = (TextView) rootView.findViewById(R.id.text);
        contentContainer = rootView.findViewById(R.id.content_container);
        buttonsContainer = (LinearLayout) rootView.findViewById(R.id.buttons_container);
        webContainer = (WebView) rootView.findViewById(R.id.web_container);
        mapView = (MapView) rootView.findViewById(R.id.map);
        MapsInitializer.initialize(getActivity());

        initContent(savedInstanceState);
        initActionButton();

        EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_QR, resultType.toString(), 0L);

        return rootView;
    }

    @Override
    public void onResume() {
        super.onResume();
        ActionBar actionBar = getActivity().getActionBar();
        actionBar.setBackgroundDrawable(actionBarBg);
        actionBar.setTitle(context.getString(R.string.qr_code_fragment_title));
        actionBar.setHomeButtonEnabled(true);
        actionBar.setDisplayHomeAsUpEnabled(true);

        actionBar.show();
        if (resultType == ParsedResultType.GEO) {
            mapView.onResume();
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        outState.putString(ACTION_BAR_TITLE, externalTitle);
        super.onSaveInstanceState(outState);
    }

    @Override
    public void onPause() {
        super.onPause();
        if (resultType == ParsedResultType.GEO) {
            mapView.onPause();
        }
    }

    @Override
    public void onStop() {
        ActionBar actionBar = getActivity().getActionBar();
        actionBarBg.setAlpha(255);
        actionBar.setBackgroundDrawable(actionBarBg);
        actionBar.setHomeButtonEnabled(true);
        actionBar.setDisplayHomeAsUpEnabled(true);
        actionBar.setTitle(externalTitle);
        actionBar.hide();

        barcodeDetector.resetBarcodeDetectionTimer();
        ((SnappingFragment) getFragmentManager()
                .findFragmentById(R.id.snapping_fragment)).setCameraEnabled(true);
        super.onStop();
    }

    @Override
    public void onLowMemory() {
        super.onLowMemory();
        if (resultType == ParsedResultType.GEO) {
            mapView.onLowMemory();
        }
    }

    @Override
    public void onDestroy() {
        if (resultType == ParsedResultType.GEO) {
            mapView.onDestroy();
        }
        super.onDestroy();
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        menu.clear();
        inflater.inflate(R.menu.barcode_menu, menu);
        super.onCreateOptionsMenu(menu, inflater);
    }

    private void initContent(Bundle savedInstanceState) {
        String displayTitle = barcodeHandler.getDisplayTitle();
        title.setVisibility(TextUtils.isEmpty(displayTitle) ? View.GONE : View.VISIBLE);
        title.setText(displayTitle);

        String displaySubtitle = barcodeHandler.getDisplaySubtitle();
        subTitle.setVisibility(TextUtils.isEmpty(displaySubtitle) ? View.GONE : View.VISIBLE);
        subTitle.setText(displaySubtitle);

        String displayText = barcodeHandler.getDisplayText();
        contentText.setVisibility(TextUtils.isEmpty(displayText) ? View.GONE : View.VISIBLE);
        contentText.setText(displayText);

        codeTypeImage.setImageResource(barcodeHandler.getDisplayIconRes());

        switch (resultType) {
            case ADDRESSBOOK:
            case EMAIL_ADDRESS:
            case PRODUCT:
            case WIFI:
            case TEL:
            case SMS:
            case CALENDAR:
            case ISBN:
                break;
            case URI:
                contentContainer.setVisibility(View.GONE);
                codeTypeImage.setVisibility(View.GONE);
                webContainer.setVisibility(View.VISIBLE);
                buttonsContainer.setBackgroundColor(getResources().getColor(R.color.qr_code_button_bg));

                webContainer.getSettings().setJavaScriptEnabled(true);
                webContainer.setWebViewClient(new WebViewClient());
                webContainer.setWebChromeClient(new WebChromeClient() {
                    @Override
                    public void onProgressChanged(WebView view, int newProgress) {
                        super.onProgressChanged(view, newProgress);

                        final FragmentActivity activity = getActivity();
                        if (activity == null) {
                            return;
                        }

                        if (newProgress >= 100) {
                            activity.setProgressBarVisibility(false);
                        } else {
                            activity.setProgress(100 * newProgress);
                            activity.setProgressBarVisibility(true);
                        }
                    }
                });
                webContainer.loadUrl(barcodeHandler.getDisplayTitle());
                break;
            case GEO:
                if (GooglePlayServicesUtil.isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS) {
                    mapView.setVisibility(View.VISIBLE);
                    contentContainer.setVisibility(View.GONE);
                    codeTypeImage.setVisibility(View.GONE);
                    buttonsContainer.setBackgroundColor(getResources().getColor(R.color.qr_code_button_bg));
                    mapView.onCreate(savedInstanceState);
                    map = mapView.getMap();

                    if (map == null) {
                        return;
                    }

                    GeoParsedResult geoResult = (GeoParsedResult) barcodeHandler.getResult();
                    LatLng position = new LatLng(geoResult.getLatitude(), geoResult.getLongitude());
                    map.addMarker(new MarkerOptions().position(position));

                    map.moveCamera(CameraUpdateFactory.newLatLngZoom(position, DEFAULT_MAP_ZOOM));
                }
                break;
            default:
                codeTypeImage.setVisibility(View.GONE);
                contentText.setGravity(Gravity.LEFT);
                contentText.setTextSize(TypedValue.COMPLEX_UNIT_PX,
                        getResources().getDimensionPixelSize(R.dimen.barcode_text_size));
                break;
        }
    }

    private void initActionButton() {
        buttonsContainer.setVisibility(barcodeHandler.getButtonCount() > 0 ? View.VISIBLE : View.GONE);
        for (int index = 0; index < barcodeHandler.getButtonCount(); index++) {
            CustomTypefaceTextView button = (CustomTypefaceTextView) getLayoutInflater(null).inflate(R.layout.barcode_button, null);
            button.setText(barcodeHandler.getButtonText(index));
            final int finalIndex = index;
            button.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    barcodeHandler.handleButtonPress(finalIndex);
                }
            });
            buttonsContainer.addView(button);
        }
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == R.id.copy) {
            ClipboardManager clipboard = (ClipboardManager) getActivity().getSystemService(Context.CLIPBOARD_SERVICE);
            android.content.ClipData clip = android.content.ClipData.newPlainText("", barcodeHandler.getResult().getDisplayResult());
            clipboard.setPrimaryClip(clip);
            Toast.makeText(getActivity(), getString(R.string.msg_copied_to_clipboard), Toast.LENGTH_SHORT).show();
            return true;
        } else if (item.getItemId() == R.id.share) {
            Intent sharingIntent = new Intent(Intent.ACTION_SEND);
            sharingIntent.setType("text/plain");
            sharingIntent.putExtra(Intent.EXTRA_TEXT, barcodeHandler.getResult().getDisplayResult());
            startActivity(Intent.createChooser(sharingIntent, getString(R.string.share_via_title)));
            return true;
        } else if (item.getItemId() == android.R.id.home) {
            getActivity().onBackPressed();
            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    /**
     * @return {@code true} if fragment shows {@link android.webkit.WebView} and can navigate back, {@code false} - otherwise
     */
    public boolean onBackPressed() {
        boolean canGoBack = resultType == ParsedResultType.URI && webContainer.canGoBack();
        if (canGoBack) {
            webContainer.goBack();
        }
        return canGoBack;
    }
}
