package net.doo.snap.lib.ui;

import android.content.SharedPreferences;
import android.os.Bundle;
import android.text.Editable;
import android.text.Spannable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.text.style.ImageSpan;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;
import com.google.inject.Inject;
import net.doo.snap.lib.IntentExtras;
import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.ActivityAnalytics;
import net.doo.snap.lib.smartname.SmartNameGenerator;
import net.doo.snap.lib.smartname.TermType;
import net.doo.snap.lib.smartname.TermTypeCategory;
import net.doo.snap.lib.util.ui.ViewUtils;
import net.doo.snap.lib.ui.widget.TermSpanDrawable;
import net.doo.snap.lib.ui.widget.text.CustomTypefaceTextView;
import net.doo.snap.lib.util.FileUtils;
import net.doo.snap.lib.util.smartname.SmartNameUtil;
import org.apache.commons.lang.StringUtils;
import roboguice.activity.RoboActivity;

import java.util.ArrayList;
import java.util.List;

import static net.doo.snap.lib.smartname.SmartNameGenerator.*;

/**
 * Provides smart name template builder
 */
public class SmartNameActivity extends RoboActivity {

    @Inject
    private SmartNameGenerator smartNameBuilder;
    @Inject
    private SharedPreferences preferences;
    @Inject
    private ActivityAnalytics activityAnalytics;

    private EditText nameInput;
    private ViewGroup termsContainer;
    private View clearButton;
    private List<TermType> termsList = new ArrayList<>();
    private TextView exampleTextView;
    private View saveBtn;

    private TextWatcher nameInputTextWatcher = new TextWatcher() {
        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            if (count != 0 && count != after) {
                int affectedTermsCount = StringUtils.countMatches(s.toString().substring(start, start + count), TERM_CHARACTER);
                if (affectedTermsCount > 0) {
                    int beforeTermsCount = StringUtils.countMatches(s.toString().substring(0, start), TERM_CHARACTER);
                    termsList.subList(beforeTermsCount, beforeTermsCount + affectedTermsCount).clear();
                }
            }
        }

        @Override
        public void onTextChanged(CharSequence s, int start, int before, int count) {
        }

        @Override
        public void afterTextChanged(Editable s) {
            clearButton.setVisibility(TextUtils.isEmpty(s) ? View.GONE : View.VISIBLE);
            updateExampleText();
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.smart_name_layout);

        initUi();

        initTermsCategories();

        if (savedInstanceState == null) {
            restoreStateFromPreferences();
        } else {
            restoreStateFromBundle(savedInstanceState);
        }
    }

    private void initUi() {
        termsContainer = (ViewGroup) findViewById(R.id.terms_container);
        exampleTextView = (TextView) findViewById(R.id.smart_name_example);

        nameInput = (EditText) findViewById(R.id.name_input);
        nameInput.addTextChangedListener(nameInputTextWatcher);
        nameInput.setCustomSelectionActionModeCallback(ViewUtils.DISABLED_ACTIONMODE_CALLBACK);
        nameInput.setLongClickable(false);
        nameInput.setTextIsSelectable(false);

        clearButton = findViewById(R.id.clear);
        clearButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                nameInput.setText(StringUtils.EMPTY);
            }
        });

        findViewById(R.id.reset).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                resetTemplate();
            }
        });

        saveBtn = findViewById(R.id.save);
        saveBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                CharSequence exampleText = exampleTextView.getText();
                if (TextUtils.isEmpty(exampleText)) {
                    return;
                } else if (FileUtils.isFileNameSafe(exampleText)) {
                    saveSmartNameInfo();
                    finish();
                } else {
                    Toast.makeText(SmartNameActivity.this, R.string.unsupported_file_name_msg, Toast.LENGTH_LONG).show();
                }
            }
        });
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        List<String> typesList = new ArrayList<>();
        for (TermType termType : termsList) {
            typesList.add(String.valueOf(termType.getId()));
        }
        outState.putString(IntentExtras.SMART_NAME_TEMPLATE, nameInput.getText().toString());
        outState.putString(IntentExtras.SMART_NAME_TERMS_TYPES, StringUtils.join(typesList, SEPARATOR));
        outState.putString(IntentExtras.SMART_NAME_EXAMPLE, exampleTextView.getText().toString());
        super.onSaveInstanceState(outState);
    }

    private void restoreStateFromPreferences() {
        String template = preferences.getString(PreferencesConstants.SMART_NAME_TEMPLATE, null);
        String termsTypes = preferences.getString(PreferencesConstants.SMART_NAME_TERMS_TYPES, null);
        restoreState(template, termsTypes);
    }

    private void restoreStateFromBundle(Bundle savedInstanceState) {
        String template = savedInstanceState.getString(IntentExtras.SMART_NAME_TEMPLATE, null);
        String termsTypes = savedInstanceState.getString(IntentExtras.SMART_NAME_TERMS_TYPES, null);
        restoreState(template, termsTypes);
    }

    private void restoreState(String template, String termsTypes) {
        if (StringUtils.isEmpty(template)) {
            resetTemplate();
            return;
        }
        if (!StringUtils.isEmpty(termsTypes)) {
            termsList.addAll(SmartNameUtil.convertTermsFromPreferencesFormat(termsTypes));

            setTemplateWithSpans(template);
        } else {
            nameInput.setText(template);
        }
    }

    private void resetTemplate() {
        nameInput.setText(StringUtils.EMPTY);
        termsList.addAll(DEFAULT_TERMS);
        setTemplateWithSpans(DEFAULT_TEMPLATE);
    }

    private void setTemplateWithSpans(String template) {
        nameInput.setText(template);
        for (int i = 0, j = 0; i < template.length(); i++) {
            if (TERM_CHARACTER.charAt(0) == template.charAt(i)) {
                addSpan(termsList.get(j++), i, i + 1);
            }
        }
    }

    private void saveSmartNameInfo() {
        List<String> typesList = new ArrayList<>();
        for (TermType termType : termsList) {
            typesList.add(String.valueOf(termType.getId()));
        }
        preferences.edit()
                .putString(PreferencesConstants.SMART_NAME_TEMPLATE, nameInput.getText().toString())
                .putString(PreferencesConstants.SMART_NAME_TERMS_TYPES, StringUtils.join(typesList, SEPARATOR))
                .putString(PreferencesConstants.SMART_NAME_EXAMPLE, exampleTextView.getText().toString())
                .apply();
    }

    private void addSpan(TermType termType, int start, int end) {
        TermSpanDrawable spanDrawable = new TermSpanDrawable(this, getString(termType.getDisplayNameResId()));
        spanDrawable.setBounds(0, 0, spanDrawable.getIntrinsicWidth(), spanDrawable.getIntrinsicHeight());
        ImageSpan span = new ImageSpan(spanDrawable);
        nameInput.getText().setSpan(span, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
        nameInput.setSelection(end);

        updateExampleText();
    }

    private void addTerm(TermType termType) {
        int selectionStart = nameInput.getSelectionStart();
        int affectedTermsCount = StringUtils.countMatches(nameInput.getText().toString().substring(0, selectionStart), TERM_CHARACTER);
        termsList.add(affectedTermsCount, termType);
        nameInput.getText().insert(selectionStart, TERM_CHARACTER);
        addSpan(termType, selectionStart, selectionStart + 1);
    }

    private void initTermsCategories() {
        for (TermTypeCategory category : TermTypeCategory.values()) {
            LinearLayout categoryView = (LinearLayout) View.inflate(this, R.layout.smart_name_terms_category_layout, null);
            ((CustomTypefaceTextView) categoryView.findViewById(R.id.category_title)).setText(category.getDisplayNameResId());
            initTerms(category, categoryView);
            termsContainer.addView(categoryView);
        }
    }

    private void initTerms(TermTypeCategory category, LinearLayout categoryView) {
        for (final TermType termType : category.getTermTypes()) {
            CustomTypefaceTextView termView = (CustomTypefaceTextView) View.inflate(this, R.layout.smart_name_term_layout, null);
            termView.setText(termType.getDisplayNameResId());
            termView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    addTerm(termType);
                }
            });
            LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
            params.setMargins(getResources().getDimensionPixelSize(R.dimen.smart_name_term_margin), 0, getResources().getDimensionPixelSize(R.dimen.smart_name_term_margin), 0);
            categoryView.addView(termView, params);
        }
    }

    private void updateExampleText() {
        String nameInputText = nameInput.getText().toString();
        String nameExample = smartNameBuilder.generateDocumentName(nameInputText, termsList);
        exampleTextView.setText(nameExample);
    }

}
