package net.doo.snap.lib.snap.edit;

import android.animation.Animator;
import android.app.Activity;
import android.support.v4.app.Fragment;
import android.view.View;
import android.view.ViewPropertyAnimator;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageStoreStrategy;
import net.doo.snap.lib.snap.PreviewFragmentPagerAdapter;
import net.doo.snap.lib.snap.edit.events.DeletePageEvent;
import net.doo.snap.lib.snap.edit.util.PageEditorHelper;
import net.doo.snap.lib.util.ui.UndoToast;
import net.doo.snap.lib.util.ui.ViewUtils;
import net.doo.snap.lib.ui.widget.ViewPager;

import roboguice.event.Observes;

/**
 * Observes {@link net.doo.snap.lib.snap.edit.events.DeletePageEvent} and deletes current page
 */
public class DeletePageEditor implements PagesEditor, UndoToast.OnUndoListener {

    private static final int SLIDE_DOWN_FACTOR = 10;

    private Activity activity;
    private EditLock editLock;
    private PageStoreStrategy strategy;

    private ViewPager pager;
    private PreviewFragmentPagerAdapter adapter;
    private PageEditorHelper editorHelper;

    private int lastDeletedItemPosition;
    private Page lastDeletedPage;

    @Inject
    public DeletePageEditor(Activity activity, EditLock editLock, PageStoreStrategy strategy) {
        this.activity = activity;
        this.editLock = editLock;
        this.strategy = strategy;
    }

    @Override
    public void attachViewPager(ViewPager viewPager, PreviewFragmentPagerAdapter adapter) {
        this.pager = viewPager;
        this.adapter = adapter;

        editorHelper = new PageEditorHelper(viewPager, adapter);
    }

    @SuppressWarnings("unused")
    public void onPageDeletion(@Observes DeletePageEvent event) {
        if (editLock.isEditLocked()) {
            return;
        }

        int currentItem = pager.getCurrentItem();
        if (adapter.getPagesCount() == 0 || currentItem == adapter.getCount() - 1) {
            return;
        }

        buildDeletionAnimator(currentItem).start();
    }

    private ViewPropertyAnimator buildDeletionAnimator(final int itemToDelete) {
        final Page pageToDelete = adapter.getDocumentDraft().getPage(itemToDelete);
        return deletePageAnimator()
                .setListener(new ViewUtils.DefaultAnimationListener() {

                    @Override
                    public void onAnimationStart(Animator animation) {
                        editLock.lockEdit();
                    }

                    @Override
                    public void onAnimationFinished(Animator animation) {
                        lastDeletedPage = pageToDelete;
                        lastDeletedItemPosition = itemToDelete;

                        if (adapter.getPagesCount() > 1 && pager.getCurrentItem() == itemToDelete) {
                            editorHelper.deleteCurrentPage(new ViewUtils.DefaultAnimationListener() {
                                @Override
                                public void onAnimationFinished(Animator animation) {
                                    editLock.unlockEdit();

                                    showUndo();
                                }
                            });
                        } else {
                            adapter.deletePage(pageToDelete);
                            editLock.unlockEdit();

                            showUndo();
                        }
                    }
                });
    }

    private ViewPropertyAnimator deletePageAnimator() {
        final Fragment fragment = adapter.getFragment(pager, pager.getCurrentItem());
        final View view = fragment.getView();

        return view.animate()
                .alpha(ViewUtils.ALPHA_TRANSPARENT)
                .translationYBy(pager.getHeight() / SLIDE_DOWN_FACTOR);
    }

    private void showUndo() {
        new UndoToast(activity)
                .listener(this)
                .message(R.string.page_deleted_toast)
                .duration(UndoToast.DURATION_NORMAL_MS)
                .show();
    }

    @Override
    public void onUndo() {
        editorHelper.addPageToPosition(lastDeletedPage, lastDeletedItemPosition, new ViewUtils.DefaultAnimationListener() {
            @Override
            public void onAnimationStart(Animator animator) {
                editLock.lockEdit();
            }

            @Override
            public void onAnimationFinished(Animator animator) {
                lastDeletedPage = null;
                editLock.unlockEdit();
            }
        });
    }
}
