package net.doo.snap.lib.ui;

import android.animation.Animator;
import android.annotation.TargetApi;
import android.app.ActionBar;
import android.app.Activity;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.DataSetObserver;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.SystemClock;
import android.support.v4.app.Fragment;
import android.text.TextUtils;
import android.view.DragEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.DecelerateInterpolator;
import android.widget.ImageView;

import com.google.inject.Inject;

import net.doo.snap.lib.IntentExtras;
import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.AnalyticsConst;
import net.doo.snap.lib.analytics.EasyTrackerWrapper;
import net.doo.snap.lib.edit.EditPolygonActivity;
import net.doo.snap.lib.genuineness.GenuinenessChecker;
import net.doo.snap.lib.persistence.DocumentDraft;
import net.doo.snap.lib.persistence.OptimizationType;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageStoreStrategy;
import net.doo.snap.lib.persistence.PictureProcessor;
import net.doo.snap.lib.persistence.RotationType;
import net.doo.snap.lib.smartname.SmartNameGenerator;
import net.doo.snap.lib.snap.PreviewFragmentPagerAdapter;
import net.doo.snap.lib.snap.SaveModeFragment;
import net.doo.snap.lib.snap.camera.CameraActivity;
import net.doo.snap.lib.snap.camera.CameraPreviewFragment;
import net.doo.snap.lib.snap.edit.ChangeFilterFragment;
import net.doo.snap.lib.snap.edit.DeletePageEditor;
import net.doo.snap.lib.snap.edit.EditLock;
import net.doo.snap.lib.snap.edit.PagesEditor;
import net.doo.snap.lib.snap.edit.RenameDocumentFragment;
import net.doo.snap.lib.snap.edit.RotatePageEditor;
import net.doo.snap.lib.snap.edit.drag.PagesRearranger;
import net.doo.snap.lib.snap.edit.events.DeletePageEvent;
import net.doo.snap.lib.snap.edit.events.DocumentRenamedEvent;
import net.doo.snap.lib.snap.edit.events.RetakePageEvent;
import net.doo.snap.lib.snap.edit.events.RotatePageEvent;
import net.doo.snap.lib.snap.event.CheckGenuinenessEvent;
import net.doo.snap.lib.snap.event.PageOptimizationTypeChangedEvent;
import net.doo.snap.lib.snap.event.PagesChangedEvent;
import net.doo.snap.lib.snap.event.PictureProcessingStatusChangedEvent;
import net.doo.snap.lib.snap.event.PictureTakenEvent;
import net.doo.snap.lib.snap.event.ProcessDocumentsEvent;
import net.doo.snap.lib.snap.event.SaveDocumentEvent;
import net.doo.snap.lib.snap.preview.ImagePreviewFragment;
import net.doo.snap.lib.snap.preview.zoom.PrepareZoomEvent;
import net.doo.snap.lib.snap.preview.zoom.ZoomingFinishedEvent;
import net.doo.snap.lib.ui.events.OnPageLongPressedEvent;
import net.doo.snap.lib.ui.tutorial.AddNewPageTutorialFragment;
import net.doo.snap.lib.ui.tutorial.RearrangementTutorialFragment;
import net.doo.snap.lib.ui.widget.MultiStateImageButton;
import net.doo.snap.lib.ui.widget.ViewPager;
import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.log.DebugLog;
import net.doo.snap.lib.util.ui.ViewUtils;

import java.io.IOException;

import roboguice.activity.RoboFragmentActivity;
import roboguice.activity.event.OnActivityResultEvent;
import roboguice.event.EventManager;
import roboguice.event.EventThread;
import roboguice.event.Observes;
import roboguice.fragment.RoboFragment;

/**
 * Main fragment of snapping workflow.
 * Shows {@link net.doo.snap.lib.snap.camera.CameraPreviewFragment} for snapping
 * and {@link net.doo.snap.lib.snap.preview.ImagePreviewFragment} for each snapped image
 * in {@link net.doo.snap.lib.ui.widget.ViewPager}
 */
public class SnappingFragment extends RoboFragment {

    private static final long RETAKE_PHOTO_PLACEHOLDER_SCALE_DURATION_MILLIS = 300L;

    public static final String DOCUMENT_DRAFT = "DOCUMENT_DRAFT";
    public static final String DOCUMENT_NAME_UPDATES = "DOCUMENT_NAME_UPDATES";

    public static final int BITMAP_SCALE_RATIO = 3;

    /**
     * Progress of fully opened page
     */
    public static final float PAGE_OPENED_PROGRESS = 1f;

    private static final float CAMERA_PREVIEW_PRESCALE = 1.4f;
    private static final float CAMERA_PREVIEW_SCALE = 0.7f;

    public static final int EDIT_ACTIVITY_REQUEST_CODE = 18;
    public static final int RETAKE_PHOTO_REQUEST_CODE = 19;
    public static final String SAVE_TYPE_FRAGMENT_TAG = "SAVE_TYPE_FRAGMENT_TAG";
    public static final String RENAME_DOCUMENT_FRAGMENT_TAG = "RENAME_DOCUMENT_FRAGMENT_TAG";

    private static final int ADD_NEW_PAGE_TUTORIAL_PAGES_COUNT = 1;
    private static final int REARRANGEMENT_TUTORIAL_PAGES_COUNT = 3;
    private static final float PAGE_TRANSFORMATION_DELTA = 0.005f;
    private static final String CHANGE_FILTER_FRAGMENT_TAG = "CHANGE_FILTER_FRAGMENT_TAG";

    @Inject
    private EventManager eventManager;
    @Inject
    private PreviewFragmentPagerAdapter adapter;
    @Inject
    private PageStoreStrategy pageStoreStrategy;
    @Inject
    private PictureProcessor pictureProcessor;
    @Inject
    private EditLock editLock;
    @Inject
    private SharedPreferences preferences;
    @Inject
    private BitmapLruCache bitmapLruCache;
    @Inject
    private GenuinenessChecker checker;
    @Inject
    private SmartNameGenerator smartNameBuilder;

    //page editors
    @Inject
    private PagesRearranger pagesRearranger;
    @Inject
    private DeletePageEditor deletePageEditor;
    @Inject
    private RotatePageEditor rotatePageEditor;

    private final Handler handler = new Handler();

    private View rootView;
    private ViewPager pager;
    private View bottomBar;
    private View retakeCameraPlaceholder;
    private ImageView snappedPreview;

    private Drawable actionBarBackground;

    private boolean animatingRetakePhoto = false;
    private final DataSetObserver pagesObserver = new DataSetObserver() {
        @Override
        public void onChanged() {
            super.onChanged();
            notifyPagesChanged();
            getActivity().invalidateOptionsMenu();
        }

        @Override
        public void onInvalidated() {
            super.onInvalidated();
            notifyPagesChanged();
            getActivity().invalidateOptionsMenu();
        }
    };

    private MultiStateImageButton filterButton;
    private boolean docNameUpdatesEnabled = true;

    public SnappingFragment() {
        setHasOptionsMenu(true);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        rootView = inflater.inflate(R.layout.snapping_screen, container, false);

        actionBarBackground = getResources().getDrawable(R.color.action_bar_bg);
        getSupportActionBar().setBackgroundDrawable(actionBarBackground);

        restoreState(savedInstanceState);
        pager = (ViewPager) rootView.findViewById(R.id.pager);
        bottomBar = rootView.findViewById(R.id.bottom_bar);
        retakeCameraPlaceholder = rootView.findViewById(R.id.placeholder);
        snappedPreview = (ImageView) rootView.findViewById(R.id.snapped_preview);

        pager = (ViewPager) rootView.findViewById(R.id.pager);
        pager.setPageMargin(getResources().getDimensionPixelSize(R.dimen.snapped_page_margin));
        pager.setPageTransformer(false, new ViewPager.PageTransformer() {
            @Override
            public void transformPage(View page, float position) {
                if (adapter.isViewFromObject(page, adapter.getCameraFragment())) {
                    if (position < PAGE_TRANSFORMATION_DELTA) {
                        position = 0f;
                    }

                    float cameraPosition = (PreviewFragmentPagerAdapter.PREVIEW_PAGE_WIDTH - position) / PreviewFragmentPagerAdapter.PREVIEW_PAGE_WIDTH;
                    if (cameraPosition < 0f) {
                        cameraPosition = 0f;
                    }

                    onCameraPageTransformation(
                            page,
                            cameraPosition
                    );
                }
            }
        });

        pager.setAdapter(adapter);
        pager.setOnPageChangeListener(new ViewPager.OnPageChangeListener() {
            @Override
            public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
            }

            @Override
            public void onPageSelected(int position) {
                final boolean cameraVisible = currentPageIsCamera();
                eventManager.fire(new PagesChangedEvent(adapter.getDocumentDraft(), cameraVisible));

                if (!cameraVisible) {
                    Page currentPage = adapter.getDocumentDraft().getPage(position);
                    filterButton.setCurrentState(currentPage.getOptimizationType().ordinal());
                }
            }

            @Override
            public void onPageScrollStateChanged(int state) {
            }
        });
        adapter.registerDataSetObserver(pagesObserver);

        attachPageEditors();

        filterButton = (MultiStateImageButton) rootView.findViewById(R.id.filter);
        initFilterButton(filterButton);

        View trash = rootView.findViewById(R.id.trash);
        trash.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (currentPageIsCamera()) {
                    return;
                }

                eventManager.fire(new DeletePageEvent());

                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_BUTTON_PRESS, "delete_page_button", 0L);
            }
        });

        View rotate = rootView.findViewById(R.id.rotate);
        rotate.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (currentPageIsCamera()) {
                    return;
                }

                eventManager.fire(new RotatePageEvent());

                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_BUTTON_PRESS, "rotate_page_button", 0L);
            }
        });

        View crop = rootView.findViewById(R.id.crop);
        crop.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (currentPageIsCamera()) {
                    return;
                }

                int position = pager.getCurrentItem();
                Intent intent = new Intent(getActivity(), EditPolygonActivity.class);
                intent.putExtra(EditPolygonActivity.PAGE, adapter.getDocumentDraft().getPage(position));
                startActivityForResult(intent, EDIT_ACTIVITY_REQUEST_CODE);
            }
        });

        setupDragListener();

        return rootView;
    }

    private void initFilterButton(final MultiStateImageButton filter) {
        filter
                .addState(OptimizationType.NONE.ordinal(), R.drawable.ui_reviewsnapping_ico_color)
                .addState(OptimizationType.COLOR_ENHANCED.ordinal(), R.drawable.ui_reviewsnapping_ico_nofilter)
                .addState(OptimizationType.BLACK_AND_WHITE.ordinal(), R.drawable.ui_reviewsnapping_ico_bw)
                .addState(OptimizationType.GRAYSCALE.ordinal(), R.drawable.ui_reviewsnapping_ico_grey);

        final int currentItem = pager.getCurrentItem();
        final boolean cameraVisible = (currentItem == adapter.getCount() - 1);
        if (!cameraVisible) {
            Page currentPage = adapter.getDocumentDraft().getPage(currentItem);
            filter.setCurrentState(currentPage.getOptimizationType().ordinal());
        }

        filter.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View view, MotionEvent event) {
                return editLock.isEditLocked();
            }
        });

        filter.setOnStateChangedListener(new MultiStateImageButton.OnStateChangedListener() {
            @Override
            public boolean onStateChange(int newState) {
                if (editLock.isEditLocked() || currentPageIsCamera()) {
                    return false;
                }

                Page currentPage = adapter.getDocumentDraft().getPage(pager.getCurrentItem());
                ChangeFilterFragment.newInstance(currentPage, filter, R.id.pager).show(getFragmentManager(), CHANGE_FILTER_FRAGMENT_TAG);

                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_BUTTON_PRESS, "image_filter_button", (long) newState);

                return false;
            }
        });
    }

    @SuppressWarnings("unused")
    private void onPageOptimizationTypeChanged(@Observes PageOptimizationTypeChangedEvent event) {
        OptimizationType optimizationType = event.getOptimizationType();

        int currentItem = pager.getCurrentItem();
        Page currentPage;
        try {
            currentPage = adapter.getDocumentDraft().getPage(currentItem);
        } catch (DocumentDraft.NoPageByIndexException e) {
            DebugLog.logException(e);
            return;
        }
        currentPage.setOptimizationType(optimizationType);

        filterButton.setCurrentState(optimizationType.ordinal());

        preferences.edit().putInt(PreferencesConstants.LAST_USED_FILTER, optimizationType.getCode()).commit();
    }

    private void notifyPagesChanged() {
        eventManager.fire(new PagesChangedEvent(adapter.getDocumentDraft(), pager.getCurrentItem() == adapter.getCount() - 1));
    }

    @Override
    public void onResume() {
        super.onResume();

        repairCameraState();
        notifyPagesChanged();
    }

    private void repairCameraState() {
        /*
        FIXME
        Workaround for ViewPager issue
         */

        pager.post(new Runnable() {
            @Override
            public void run() {
                final int item = pager.getCurrentItem();
                pager.setCurrentItem(adapter.getCount() - 1, false);
                pager.post(new Runnable() {
                    @Override
                    public void run() {
                        pager.setCurrentItem(item, false);
                    }
                });
            }
        });
    }

    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    private void setupDragListener() {
        class DragListener implements View.OnDragListener {

            @Override
            public boolean onDrag(View v, DragEvent event) {
                switch (event.getAction()) {
                    case DragEvent.ACTION_DRAG_STARTED:
                        hideBars();
                        break;
                    case DragEvent.ACTION_DRAG_ENDED:
                        if (!animatingRetakePhoto) {
                            showBars();
                        }
                        break;
                }

                return false;
            }
        }

        rootView.setOnDragListener(new DragListener());
    }

    private void attachPageEditors() {
        PagesEditor[] editors = {
                pagesRearranger,
                deletePageEditor,
                rotatePageEditor
        };

        for (PagesEditor editor : editors) {
            editor.attachViewPager(pager, adapter);
        }
    }

    /**
     * Invoked when new transformation for camera page should be applied
     *
     * @param cameraView {@link android.view.View} that contains camera's UI
     * @param progress   open progress of the camera
     */
    protected void onCameraPageTransformation(View cameraView, float progress) {
        updateActionBarStatus(progress);

        adapter.getCameraFragment().setOpenProgress(progress);

        final float scale = Math.max(CAMERA_PREVIEW_SCALE, progress);
        cameraView.setScaleX(scale);
        cameraView.setScaleY(scale);

        final int cameraWidth = cameraView.getWidth() - 2 * pager.getPageMargin();
        cameraView.setTranslationX(
                (progress - 1f) * (cameraWidth - cameraWidth * scale) / 2f
        );
    }

    private void restoreState(Bundle savedInstanceState) {
        if (savedInstanceState == null) {
            return;
        }

        adapter.setDocumentDraft((DocumentDraft) savedInstanceState.getParcelable(DOCUMENT_DRAFT));
        updateTitle();
        docNameUpdatesEnabled = savedInstanceState.getBoolean(DOCUMENT_NAME_UPDATES);
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putParcelable(DOCUMENT_DRAFT, adapter.getDocumentDraft());
        outState.putBoolean(DOCUMENT_NAME_UPDATES, docNameUpdatesEnabled);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        switch (requestCode) {
            case EDIT_ACTIVITY_REQUEST_CODE:
                handleEditActivityResult(resultCode, data);
                eventManager.fire(new OnActivityResultEvent(requestCode, resultCode, data));
                break;
            case RETAKE_PHOTO_REQUEST_CODE:
                handleRetakePhotoResult(resultCode, data);
                break;
        }
    }

    private void handleEditActivityResult(int resultCode, Intent data) {
        if (resultCode != Activity.RESULT_OK) {
            return;
        }

        Page resultPage = data.getParcelableExtra(EditPolygonActivity.PAGE);

        int currentItem = pager.getCurrentItem();
        Page page;
        try {
            page = adapter.getDocumentDraft().getPage(currentItem);
        } catch (DocumentDraft.NoPageByIndexException e) {
            DebugLog.logException(e);
            return;
        }

        if (resultPage != null && page != null) {
            page.setPolygon(resultPage.getPolygon());
        }
    }

    private void handleRetakePhotoResult(int resultCode, Intent data) {
        retakeCameraPlaceholder.setVisibility(View.GONE);
        showBars();

        if (resultCode != Activity.RESULT_OK) {
            return;
        }

        final Page page = data.getParcelableExtra(CameraActivity.RESULT_PAGE);
        final int position = data.getIntExtra(CameraActivity.PAGE_POSITION, 0);

        pager.post(new Runnable() {
            @Override
            public void run() {
                adapter.deletePage(position);
                adapter.addPage(position, page);
                pager.setCurrentItem(position);
            }
        });
    }

    @SuppressWarnings("unused")
    public void onPictureProcessingStatusChanged(@Observes(EventThread.UI) PictureProcessingStatusChangedEvent event) {
        if (event.isProcessing()) {
            pager.setEnabled(false);
        }
    }

    @SuppressWarnings("unused")
    public void onPictureTaken(@Observes PictureTakenEvent event) {
        final long pictureProcessingStart = SystemClock.elapsedRealtime();

        try {
            final Page page = pictureProcessor.processBytes(event.getImage(), event.getImageOrientation());

            final String path = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.OPTIMIZED_PREVIEW).getPath();

            final Bitmap cached = bitmapLruCache.get(path);
            final Bitmap optimized = (cached != null)
                    ? cached
                    : BitmapFactory.decodeFile(path);
            final Bitmap scaledBitmap = Bitmap.createScaledBitmap(optimized, optimized.getWidth() / BITMAP_SCALE_RATIO, optimized.getHeight() / BITMAP_SCALE_RATIO, true);

            handler.post(new Runnable() {
                @Override
                public void run() {
                    if (!isAdded()) {
                        return;
                    }

                    addSnappedPage(page, optimized, scaledBitmap);
                    if (docNameUpdatesEnabled) {
                        adapter.getDocumentDraft().setDocumentName(smartNameBuilder.generateDocumentName());
                        updateTitle();
                    }
                }
            });
        } catch (IOException e) {
            DebugLog.logException(e);

            handler.post(new Runnable() {
                @Override
                public void run() {
                    resetSnappedPreview();

                    pager.setEnabled(true);
                    pager.setCurrentItem(adapter.getCount() - 1, false);
                    adapter.getCameraFragment().startCameraPreview();
                }
            });
        }

        final long processingDuration = SystemClock.elapsedRealtime() - pictureProcessingStart;
        EasyTrackerWrapper.sendTiming(AnalyticsConst.TIMING_SNAPPING, processingDuration, "saving_snapped_page", Build.MODEL);
    }

    private void addSnappedPage(final Page page, Bitmap optimizedBitmap, final Bitmap scaledBitmap) {
        View camera = adapter.getCameraFragment().getView();
        final int cameraWidth = camera.getWidth() - 2 * pager.getPageMargin();

        page.getParameters().putParcelable(ImagePreviewFragment.PRESERVED_BITMAP, optimizedBitmap);

        snappedPreview.setImageBitmap(scaledBitmap);
        snappedPreview.setRotation(page.getRotationType().getDegrees());
        final float scale = calculateRotationScale(page.getRotationType());
        snappedPreview.setScaleX(scale);
        snappedPreview.setScaleY(scale);

        snappedPreview.animate()
                .setListener(new ViewUtils.DefaultAnimationListener() {
                    @Override
                    public void onAnimationStart(android.animation.Animator animator) {
                        snappedPreview.setVisibility(View.VISIBLE);
                        ViewUtils.saveViewTransformation(snappedPreview);
                    }

                    @Override
                    public void onAnimationFinished(android.animation.Animator animator) {
                        /*
                        On some devices (so far detected on Android 4.0.4) animation callback
                        for new animation, launched from onAnimationFinished method, fires twice.
                        Therefore, we're postponing creation of new animation to avoid this problem.
                         */
                        handler.post(new Runnable() {
                            @Override
                            public void run() {
                                snappedPreview.animate()
                                        .scaleX(scale)
                                        .scaleY(scale)
                                        .setListener(new ViewUtils.DefaultAnimationListener() {
                                            @Override
                                            public void onAnimationFinished(Animator animator) {
                                                if (!isAdded()) {
                                                    return;
                                                }

                                                resetSnappedPreview();
                                                insertPage(page);
                                                scaledBitmap.recycle();
                                            }
                                        }).start();
                            }
                        });
                    }
                })
                .scaleX(scale * CAMERA_PREVIEW_PRESCALE)
                .scaleY(scale * CAMERA_PREVIEW_PRESCALE)
                .start();

        camera.animate()
                .scaleX(CAMERA_PREVIEW_SCALE)
                .scaleY(CAMERA_PREVIEW_SCALE)
                .translationX(cameraWidth * (1f + CAMERA_PREVIEW_SCALE) / 2f)
                .start();
    }

    private void resetSnappedPreview() {
        snappedPreview.setImageDrawable(null);
        snappedPreview.setVisibility(View.GONE);
        ViewUtils.restoreViewTransformation(snappedPreview);
    }

    private float calculateRotationScale(RotationType rotationType) {
        float pageInnerMargin = getResources().getDimension(R.dimen.page_inner_margin);
        float actionBarHeight = getActivity() != null ? getSupportActionBar().getHeight() : 0f;
        float bottomBarHeight = actionBarHeight;
        float dropTargetWidth = getResources().getDimension(R.dimen.page_drop_target_width);

        final float availableWidth = getView().getWidth() * CAMERA_PREVIEW_SCALE
                - 2f * dropTargetWidth;

        final float availableHeight = getView().getHeight()
                - 2f * pageInnerMargin
                - actionBarHeight
                - bottomBarHeight;

        if (snappedPreview.getDrawable() == null) {
            return ViewUtils.SCALE_DEFAULT;
        }

        if (rotationType == RotationType.ROTATION_0
                || rotationType == RotationType.ROTATION_180
                || rotationType == RotationType.ROTATION_360) {
            return Math.min(
                    availableWidth / snappedPreview.getDrawable().getIntrinsicWidth(),
                    availableHeight / snappedPreview.getDrawable().getIntrinsicHeight()
            );
        }

        return Math.min(
                availableWidth / snappedPreview.getDrawable().getIntrinsicHeight(),
                availableHeight / snappedPreview.getDrawable().getIntrinsicWidth()
        );
    }

    private void insertPage(Page page) {
        pager.setEnabled(true);
        adapter.addPage(page);

        final int pagesCount = adapter.getPagesCount();

        switch (pagesCount) {
            case ADD_NEW_PAGE_TUTORIAL_PAGES_COUNT:
                AddNewPageTutorialFragment.newInstance().showOnce(getActivity(), preferences);
                break;
            case REARRANGEMENT_TUTORIAL_PAGES_COUNT:
                RearrangementTutorialFragment.newInstance().showOnce(getActivity(), preferences);
                break;
        }

        pager.setCurrentItem(adapter.getCount() - 2, false);
        adapter.getCameraFragment().startCameraPreview();
    }

    @SuppressWarnings("unused")
    public void onPageLongPressed(@Observes OnPageLongPressedEvent event) {
        final int pagePosition = adapter.getPagePosition(event.getDraggedPage());

        if (pagePosition != pager.getCurrentItem()) {
            return;
        }

        pagesRearranger.startRearrangement(
                event.getX(),
                event.getY(),
                event.getView(),
                event.getDraggedPage(),
                pagePosition,
                event.getDraggedFragment(),
                event.getPreservedBitmap()
        );
    }

    @SuppressWarnings("unused")
    public void onRetakePage(@Observes RetakePageEvent event) {
        if (event.shouldAnimate()) {
            retakePhotoWithAnimation(event.getPagePosition(), event.getDropX(), event.getDropY());
        } else {
            retakePhoto(event.getPagePosition());
        }
    }

    private void retakePhotoWithAnimation(final int pagePosition, float dropX, float dropY) {
        retakeCameraPlaceholder.setScaleX(0f);
        retakeCameraPlaceholder.setScaleY(0f);
        retakeCameraPlaceholder.setTranslationX(dropX - pager.getWidth() / 2f);
        retakeCameraPlaceholder.setTranslationY(dropY - pager.getHeight() / 2f);

        retakeCameraPlaceholder.animate()
                .scaleX(ViewUtils.SCALE_DEFAULT)
                .scaleY(ViewUtils.SCALE_DEFAULT)
                .translationX(ViewUtils.TRANSLATION_DEFAULT)
                .translationY(ViewUtils.TRANSLATION_DEFAULT)
                .setInterpolator(new DecelerateInterpolator())
                .setDuration(RETAKE_PHOTO_PLACEHOLDER_SCALE_DURATION_MILLIS)
                .setListener(new ViewUtils.DefaultAnimationListener() {
                    @Override
                    public void onAnimationStart(Animator animation) {
                        retakeCameraPlaceholder.setVisibility(View.VISIBLE);
                        animatingRetakePhoto = true;
                    }

                    @Override
                    public void onAnimationFinished(Animator animation) {
                        animatingRetakePhoto = false;
                        retakePhoto(pagePosition);
                    }
                })
                .start();
    }

    private void retakePhoto(int pagePosition) {
        if (getActivity() != null) {
            Intent intent = new Intent(getActivity(), CameraActivity.class);
            intent.putExtra(CameraActivity.PAGE_POSITION, pagePosition);
            intent.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION);
            startActivityForResult(intent, RETAKE_PHOTO_REQUEST_CODE);
        }
    }

    private void updateActionBarStatus(float progress) {
        if (getFragmentManager().findFragmentByTag(BarcodeFragment.BARCODE_FRAGMENT_TAG) != null) {
            updateBarcodeActionBar();
            return;
        }

        if (pagesRearranger.isDragging() || animatingRetakePhoto) {
            return;
        }

        updateActionBarBackground(progress);

        if (progress == PAGE_OPENED_PROGRESS) {
            hideBars();
        } else {
            showBars();
        }
    }

    private void updateBarcodeActionBar() {
        updateActionBarBackground(0);

        getSupportActionBar().show();
        bottomBar.setEnabled(false);
        bottomBar.animate()
                .translationY(bottomBar.getHeight())
                .start();
    }

    private void hideBars() {
        getSupportActionBar().hide();
        bottomBar.setEnabled(false);
        bottomBar.animate()
                .translationY(bottomBar.getHeight())
                .start();
    }

    private void showBars() {
        getSupportActionBar().show();
        bottomBar.setEnabled(true);
        bottomBar.animate()
                .translationY(ViewUtils.TRANSLATION_DEFAULT)
                .start();
    }

    private void updateActionBarBackground(float progress) {
        actionBarBackground.setAlpha((int) (255f * (PAGE_OPENED_PROGRESS - progress)));
        getSupportActionBar().setBackgroundDrawable(actionBarBackground);

        bottomBar.setAlpha(PAGE_OPENED_PROGRESS - progress);

        /*
        Workaround. For some reason Android drops alpha for view background, while it shouldn't
         */
        bottomBar.getBackground().setAlpha(255);
    }

    private ActionBar getSupportActionBar() {
        return getActionBarActivity().getActionBar();
    }

    private RoboFragmentActivity getActionBarActivity() {
        return (RoboFragmentActivity) getActivity();
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        adapter.unregisterDataSetObserver(pagesObserver);
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        inflater.inflate(R.menu.snapping_menu, menu);
        final DocumentDraft documentDraft = adapter.getDocumentDraft();

        if (documentDraft.isEmpty()) {
            menu.findItem(R.id.save).setVisible(false);
            menu.findItem(R.id.rename).setVisible(false);
        }
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        DocumentDraft documentDraft = adapter.getDocumentDraft();

        /**
         * We use "if" instead of "switch" because of this:
         * http://tools.android.com/tips/non-constant-fields
         */
        if (item.getItemId() == R.id.rename) {
            Bundle args = new Bundle();
            args.putString(IntentExtras.CURRENT_NAME, documentDraft.getDocumentName());
            RenameDocumentFragment renameFragment = RenameDocumentFragment.newInstance(args);
            renameFragment.show(getFragmentManager(), RENAME_DOCUMENT_FRAGMENT_TAG);
            return true;
        } else if (item.getItemId() == R.id.save) {
            if (editLock.isEditLocked()) {
                return true;
            }

            EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_BUTTON_PRESS, "save_document", 0L);

            eventManager.fire(new CheckGenuinenessEvent(documentDraft));

            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    @SuppressWarnings("unused")
    public void onDocumentRename(@Observes DocumentRenamedEvent event) {
        docNameUpdatesEnabled = false;
        adapter.getDocumentDraft().setDocumentName(event.getNewName());
        updateTitle();

        EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_CONTENT, "document_renamed", 0L);
    }

    @SuppressWarnings("unused")
    public void onDocumentSave(@Observes ProcessDocumentsEvent event) {
        docNameUpdatesEnabled = true;
        adapter.setDocumentDraft(new DocumentDraft());
        adapter.notifyDataSetChanged();
    }

    @SuppressWarnings("unused")
    public void saveDocument(@Observes SaveDocumentEvent event) {
        DocumentDraft documentDraft = event.getDraft();
        if (documentDraft.size() <= 1) {
            eventManager.fire(new ProcessDocumentsEvent(documentDraft, true));
            EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_UI, AnalyticsConst.ACTION_BUTTON_PRESS, "save_single_document", 0L);
        } else {
            if (getFragmentManager().findFragmentByTag(SAVE_TYPE_FRAGMENT_TAG) == null) {
                SaveModeFragment.newInstance(documentDraft).show(getFragmentManager(), SAVE_TYPE_FRAGMENT_TAG);
            }
        }
    }

    /**
     * Enables or disables workflow of {@link net.doo.snap.lib.snap.camera.CameraPreviewFragment}.
     * When enabled, camera controls are shown and image processing turned on.
     * When disabled, camera controls are hidden and image processing turned off.
     *
     * @throws NullPointerException if called before {@link net.doo.snap.lib.snap.camera.CameraPreviewFragment} was
     *                              initialized
     */
    public void setCameraEnabled(boolean cameraEnabled) throws NullPointerException {
        CameraPreviewFragment cameraPreviewFragment = adapter.getCameraFragment();

        Fragment barcodeFragment = getFragmentManager().findFragmentByTag(BarcodeFragment.BARCODE_FRAGMENT_TAG);
        if (cameraPreviewFragment == null
                || (barcodeFragment != null && barcodeFragment.isVisible())) {
            return;
        }

        cameraPreviewFragment.setCameraActive(cameraEnabled);
    }

    /**
     * Moves {@link net.doo.snap.lib.ui.widget.ViewPager} to camera
     */
    public void switchToPreview() {
        pager.setCurrentItem(adapter.getCount() - 1, true);
    }

    /**
     * Updates {@link android.app.ActionBar} title using name of current {@link net.doo.snap.lib.persistence.DocumentDraft}
     */
    protected void updateTitle() {
        if (getFragmentManager().findFragmentByTag(BarcodeFragment.BARCODE_FRAGMENT_TAG) != null) {
            getSupportActionBar().setTitle(R.string.qr_code_fragment_title);
            return;
        }

        final String title = adapter.getDocumentDraft().getDocumentName();

        if (!TextUtils.isEmpty(title)) {
            getSupportActionBar().setTitle(adapter.getDocumentDraft().getDocumentName());
        }
    }

    /**
     * @return {@code true} if camera is currently selected as the main page of {@link net.doo.snap.lib.ui.SnappingFragment}
     * and not being dragged by user. {@code false} otherwise.
     */
    public boolean isCameraSelected() {
        if (adapter == null) {
            return false;
        }

        if (!currentPageIsCamera()) {
            return false;
        }

        CameraPreviewFragment cameraPreviewFragment = adapter.getCameraFragment();

        return cameraPreviewFragment != null && cameraPreviewFragment.isReady();
    }

    private boolean currentPageIsCamera() {
        return pager.getCurrentItem() == adapter.getCount() - 1;
    }

    @SuppressWarnings("unused")
    private void onStartZooming(@Observes PrepareZoomEvent event) {
        pager.setEnabled(false);
    }

    @SuppressWarnings("unused")
    private void onFinishZooming(@Observes ZoomingFinishedEvent event) {
        pager.setEnabled(true);
    }

}
