package net.doo.snap.lib.edit;

import android.app.ActionBar;
import android.app.Activity;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.PointF;
import android.os.AsyncTask;
import android.os.Bundle;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.Loader;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.ActivityAnalytics;
import net.doo.snap.lib.detector.ContourDetector;
import net.doo.snap.lib.detector.DetectionResult;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageStoreStrategy;
import net.doo.snap.lib.persistence.PictureProcessor;
import net.doo.snap.lib.util.ui.ViewUtils;
import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.bitmap.PageBitmapLoader;
import net.doo.snap.lib.util.log.DebugLog;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import roboguice.activity.RoboFragmentActivity;

/**
 * Allows to edit polygon which contains document coordinates
 */
public class EditPolygonActivity extends RoboFragmentActivity implements LoaderManager.LoaderCallbacks<Bitmap>, DrawMagnifierListener {
    public static final String PAGE = "page";
    private static final String POLYGON = "polygon";
    private static final String LOADER_PATH = "LOADER_PATH";

    private final Executor executor = Executors.newSingleThreadExecutor();

    @Inject
    private PageStoreStrategy pageStoreStrategy;
    @Inject
    private ActivityAnalytics activityAnalytics;
    @Inject
    private BitmapLruCache bitmapLruCache;

    private Bitmap bitmap;

    private EditPolygonImageView image;
    private MagnifierView magnifierView;
    private Page page;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.edit_polygon_activity);
        page = getIntent().getParcelableExtra(PAGE);
        if (page == null) {
            finish();
            return;
        }

        ActionBar actionBar = getActionBar();
        actionBar.setDisplayShowHomeEnabled(false);
        actionBar.setDisplayShowTitleEnabled(false);
        actionBar.setDisplayShowCustomEnabled(true);
        actionBar.setDisplayHomeAsUpEnabled(false);
        actionBar.setCustomView(R.layout.done);

        image = (EditPolygonImageView) findViewById(R.id.image);
        image.setRotation(page.getRotationType().getDegrees());

        magnifierView = (MagnifierView) findViewById(R.id.magnifier);
        setPolygon(savedInstanceState);

        try {
            Bundle args = buildLoaderArgs();
            getSupportLoaderManager().initLoader(0, args, this);
        } catch (IOException e) {
            DebugLog.logException(e);
        }

        View done = findViewById(R.id.done);
        done.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                page.setPolygon(image.getPolygon());
                new ImageProcessor().execute(page);
            }
        });
    }

    private Bundle buildLoaderArgs() throws IOException {
        Bundle args = new Bundle();
        final String previewPath = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.PREVIEW).getPath();
        args.putString(LOADER_PATH, previewPath);
        return args;
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putParcelableArrayList(POLYGON, (ArrayList<PointF>) image.getPolygon());
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.edit_polygon_menu, menu);
        return super.onCreateOptionsMenu(menu);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case android.R.id.home:
                onBackPressed();
                return true;
        }

        if (item.getItemId() == R.id.detect_polygon) {
            try {
                File image = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.PREVIEW);
                new DetectPolygon().executeOnExecutor(executor, image.getAbsolutePath());

                return true;
            } catch (IOException e) {
                DebugLog.logException(e);
            }

            return false;
        }

        return super.onOptionsItemSelected(item);
    }

    private void setPolygon(Bundle savedInstanceState) {
        if (savedInstanceState == null) {
            image.setPolygon(page.getPolygon());
            return;
        }

        ArrayList<PointF> polygon = savedInstanceState.getParcelableArrayList(POLYGON);
        image.setPolygon(polygon);
    }

    @Override
    public Loader<Bitmap> onCreateLoader(int i, Bundle args) {
        final String path = args.getString(LOADER_PATH);
        return new PageBitmapLoader(
                EditPolygonActivity.this,
                path,
                bitmapLruCache,
                bitmap
        );
    }

    @Override
    public void onLoadFinished(Loader<Bitmap> loader, Bitmap result) {
        image.setImageBitmap(result);
        bitmap = result;
        image.setAlpha(ViewUtils.ALPHA_TRANSPARENT);
        image.animate().alpha(ViewUtils.ALPHA_DEFAULT).start();
        magnifierView.setupMagnifier(image);
    }

    @Override
    public void onLoaderReset(Loader<Bitmap> loader) {
        bitmap = null;
        image.setImageDrawable(null);
    }

    @Override
    public void drawMagnifier(PointF zoomPoint) {
        magnifierView.drawMagnifier(zoomPoint);
    }

    @Override
    public void eraseMagnifier() {
        magnifierView.eraseMagnifier();
    }

    /**
     * Processes image and creates optimized preview
     */
    class ImageProcessor extends AsyncTask<Page, Void, Void> {

        @Override
        protected Void doInBackground(Page... params) {
            Page page = params[0];
            if (!isCancelled() && pageStoreStrategy != null) {
                ContourDetector detector = new ContourDetector();
                try {
                    PictureProcessor.generateOptimizedPreview(pageStoreStrategy, page, detector);
                } catch (IOException e) {
                    // Unable to create optimized preview
                    DebugLog.logException(e);
                }
            }

            return null;
        }

        @Override
        protected void onPostExecute(Void aVoid) {
            if (!isCancelled()) {
                Intent result = new Intent();
                result.putExtra(PAGE, page);
                setResult(Activity.RESULT_OK, result);
                finish();
            }
        }
    }

    /**
     * Detects polygon and sets it to image
     */
    class DetectPolygon extends AsyncTask<String, Void, List<PointF>> {

        @Override
        protected List<PointF> doInBackground(String... params) {
            List<PointF> polygon = Collections.emptyList();
            String fileName = params[0];

            try {
                ContourDetector detector = new ContourDetector();
                final DetectionResult detectionResult = detector.detect(fileName);
                switch (detectionResult) {
                    case OK:
                    case OK_BUT_BAD_ANGLES:
                    case OK_BUT_TOO_SMALL:
                    case OK_BUT_BAD_ASPECT_RATIO:
                        polygon = detector.getPolygonF();
                }
            } catch (IOException e) {
                DebugLog.logException(e);
            }

            return polygon;
        }

        @Override
        protected void onPostExecute(List<PointF> polygon) {
            if (!isCancelled() && !polygon.isEmpty() && image != null) {
                image.setPolygon(polygon);
            }
        }
    }
}
