package net.doo.snap.lib.ui.widget.text;

import android.content.Context;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.TypedValue;

/**
 * Automatically scales down text if it doesn't fits the container's width. Can be used only for single
 * line text.
 */
public class AutoscaleTextView extends CustomTypefaceTextView {

    private static final float TEXT_SIZE_DELTA = 1f;

    public AutoscaleTextView(Context context) {
        super(context);
    }

    public AutoscaleTextView(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    public AutoscaleTextView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        fitText(w - getPaddingLeft() - getPaddingRight());
    }

    @Override
    protected void onTextChanged(CharSequence text, int start, int lengthBefore, int lengthAfter) {
        super.onTextChanged(text, start, lengthBefore, lengthAfter);

        fitText(getWidth() - getPaddingLeft() - getPaddingRight());
    }

    private void fitText(int width) {
        if (width <= 0) {
            return;
        }

        final CharSequence text = getText();
        if (TextUtils.isEmpty(text)) {
            return;
        }

        final TextPaint paint = getPaint();

        float diff = width - paint.measureText(text, 0, text.length());

        if (diff > 0f) {
            return;
        }

        float textSize = paint.getTextSize();
        float rightBound = textSize;
        float leftBound = 0f;

        do {
            textSize = (rightBound - leftBound) / 2f + leftBound;
            paint.setTextSize(textSize);
            diff = width - paint.measureText(text, 0, text.length());

            if (diff < 0f) {
                rightBound = textSize;
            } else {
                leftBound = textSize;
            }
        } while (Math.abs(diff) > TEXT_SIZE_DELTA && Math.abs(rightBound - leftBound) > TEXT_SIZE_DELTA);

        setTextSize(TypedValue.COMPLEX_UNIT_PX, textSize);
    }
}
