package net.doo.snap.lib.edit;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.widget.ImageView;

import net.doo.snap.lib.R;
import net.doo.snap.lib.detector.Line2D;
import net.doo.snap.lib.util.log.DebugLog;
import net.doo.snap.lib.util.snap.PolygonHelper;
import net.doo.snap.lib.util.ui.ViewUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.PriorityQueue;

/**
 * Shows image of document and polygon, which contains document coordinates
 * with ability to change polygon corners and edges
 * Magnifier is shown when polygon corners are moved
 */
public class EditPolygonImageView extends ImageView {

    private static final int CORNERS_COUNT = 4;

    private final float magneticLineTreshold;

    private Paint paint;
    private Paint handlePaint;
    private PolygonHelper polygonHelper;
    private float rotationScale = 1f;

    /**
     * Detected lines defined by two points
     */
    private final List<Line2D> horizontalLines2D = new ArrayList<>();
    private final List<Line2D> verticalLines2D = new ArrayList<>();

    /**
     * Lines calculated from {@link net.doo.snap.lib.detector.Line2D} lines above
     * Used to calculate distance to {@link finger}
     */
    private final List<Line> horizontalLines = new ArrayList<>();
    private final List<Line> verticalLines = new ArrayList<>();

    /**
     * Touch coordinates inside view
     */
    private PointF finger = new PointF();

    /**
     * polygon with 4 points and coordinates from 0 to 1
     * this polygon is used as source and output for document coordinated.
     * {@link #corners} and {@link #edges} are calculated from it to increase performance
     */
    private List<PointF> polygon;

    private List<PointF> corners;
    private List<Edge> edges;

    /**
     * offset from edge to scaled image inside {@link android.widget.ImageView}
     */
    private float offsetX;
    private float offsetY;

    private Bitmap cornerBitmap;
    private Bitmap edgeBitmap;
    private int handleSize;

    private PointF selectedCorner;
    private Edge selectedEdge;
    private PointF tmpPointA = new PointF();
    private PointF tmpPointB = new PointF();
    private float medianX;
    private float medianY;
    private PriorityQueue<PointF> pointsQueue = new PriorityQueue<PointF>(CORNERS_COUNT, new Comparator<PointF>() {
        @Override
        public int compare(PointF lhs, PointF rhs) {
            double first = Math.atan2(lhs.y - medianY, lhs.x - medianX);
            double second = Math.atan2(rhs.y - medianY, rhs.x - medianX);

            return Double.compare(first, second);
        }
    });

    private DrawMagnifierListener magnifierListener;
    private float[] imageMatrix = new float[9];

    private RectF touchRect = new RectF();

    private float polygonStrokeWidth;

    /**
     * Constructor to embed view in XML layout
     *
     * @param context
     * @param attrs
     */
    public EditPolygonImageView(Context context, AttributeSet attrs) {
        super(context, attrs);

        magneticLineTreshold = getResources().getDimension(R.dimen.magnetic_line_treshold);

        polygonStrokeWidth = getResources().getDimension(R.dimen.polygon_stroke_width);
        polygon = Collections.emptyList();

        paint = new Paint();
        paint.setColor(getResources().getColor(R.color.polygon_lines));
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeWidth(polygonStrokeWidth);
        paint.setAntiAlias(true);

        handlePaint = new Paint();
        handlePaint.setAntiAlias(true);
        handlePaint.setFilterBitmap(true);

        polygonHelper = new PolygonHelper();
        corners = new ArrayList<PointF>();
        for (int i = 0; i < CORNERS_COUNT; i++) {
            corners.add(new PointF());
        }

        edges = new ArrayList<Edge>();
        for (int i = 0; i < corners.size(); i++) {
            Edge edge = new Edge();
            edge.pointA = corners.get(i);
            edge.pointB = corners.get((i + 1) % CORNERS_COUNT);
            edge.pointFarA = corners.get((i + 3) % CORNERS_COUNT);
            edge.pointFarB = corners.get((i + 2) % CORNERS_COUNT);

            edges.add(edge);
        }

        try {
            magnifierListener = (DrawMagnifierListener) context;
        } catch (ClassCastException e) {
            // activity is not DrawMagnifierListener
            DebugLog.logException(e);
        }

        handleSize = getResources().getDimensionPixelSize(R.dimen.edit_polygon_handle_size);
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inPreferredConfig = Bitmap.Config.ARGB_8888;
        cornerBitmap = BitmapFactory.decodeResource(getResources(), R.drawable.ui_crop_corner_handle, options);
        edgeBitmap = BitmapFactory.decodeResource(getResources(), R.drawable.ui_crop_side_handle, options);
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        cornerBitmap.recycle();
        edgeBitmap.recycle();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);

        if (getDrawable() != null) {
            offsetX = 0;
            offsetY = 0;
            // Get image matrix values and place them into array
            getImageMatrix().getValues(imageMatrix);

            // Extract the scale values using the constants (if aspect ratio maintained, scaleX == scaleY)
            final float scaleX = imageMatrix[Matrix.MSCALE_X];
            final float scaleY = imageMatrix[Matrix.MSCALE_Y];

            // Get the drawable (could also get the bitmap behind the drawable and getWidth/getHeight)
            final Drawable d = getDrawable();
            final int origW = d.getIntrinsicWidth();
            final int origH = d.getIntrinsicHeight();

            // Calculate the actual dimensions
            final int actW = Math.round(origW * scaleX);
            final int actH = Math.round(origH * scaleY);

            if (actW < getMeasuredWidth()) {
                offsetX = (getMeasuredWidth() - actW) >> 1;
            }

            if (actH < getMeasuredHeight()) {
                offsetY = (getMeasuredHeight() - actH) >> 1;
            }

            // Sets scaled image dimensions to polygon helper
            polygonHelper.setImageSize(actW, actH);

            if (!polygon.isEmpty()) {
                polygonHelper.getDrawingPolygon(polygon, corners);
            }
        }
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        if (getDrawable() != null) {
            rotateAndScale();
        }
    }

    /**
     * Sets polygon containing document coordinates
     *
     * @param polygon
     */
    public void setPolygon(List<PointF> polygon) {
        if (!polygon.isEmpty()) {
            this.polygon = polygon;
            polygonHelper.getDrawingPolygon(polygon, corners);
            invalidate();
        }
    }

    /**
     * @return current polygon containing document coordinates
     */
    public List<PointF> getPolygon() {
        polygonHelper.getPolygonFromDrawingPolygon(corners, polygon);
        return polygon;
    }

    /**
     * Sets detected horizontal and vertical lines
     * @param horizontalLines
     * @param verticalLines
     */
    public void setLines(List<Line2D> horizontalLines, List<Line2D> verticalLines) {
        horizontalLines2D.clear();
        horizontalLines2D.addAll(polygonHelper.scaleLines(horizontalLines));

        verticalLines2D.clear();
        verticalLines2D.addAll(polygonHelper.scaleLines(verticalLines));

        this.horizontalLines.clear();
        this.verticalLines.clear();
        for (Line2D line2D : horizontalLines) {
            Line line = new Line();
            line.calculateLine(line2D.getStart(), line2D.getEnd());
            this.horizontalLines.add(line);
        }

        for (Line2D line2D : verticalLines) {
            Line line = new Line();
            line.calculateLine(line2D.getStart(), line2D.getEnd());
            this.verticalLines.add(line);
        }
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        if (!polygon.isEmpty()) {
            canvas.save();
            canvas.translate(offsetX, offsetY);
            for (int i = 0; i < corners.size(); i++) {
                PointF startPoint = corners.get(i);
                PointF endPoint = corners.get((i + 1) % CORNERS_COUNT);
                canvas.drawLine(startPoint.x, startPoint.y, endPoint.x, endPoint.y, paint);
            }

            if (selectedCorner != null) {
                drawHandle(canvas, selectedCorner, cornerBitmap, 0);
                canvas.save();
                canvas.translate(-offsetX, -offsetY);

                if (magnifierListener != null) {
                    magnifierListener.drawMagnifier(selectedCorner);
                }
                canvas.restore();
            } else if (selectedEdge != null) {
                drawHandle(canvas, selectedEdge.getHandle(), edgeBitmap, selectedEdge.getAngleInDegrees());
            } else {
                for (Edge edge : edges) {
                    drawHandle(canvas, edge.getHandle(), edgeBitmap, edge.getAngleInDegrees());
                }

                for (PointF point : corners) {
                    drawHandle(canvas, point, cornerBitmap, 0);
                }
            }

            canvas.restore();
        }
    }

    private void drawHandle(Canvas canvas, PointF point, Bitmap handle, float degrees) {
        canvas.save();
        if (degrees != 0) {
            canvas.rotate(degrees, point.x, point.y);
        }

        float counterScale = 1 / rotationScale;
        canvas.scale(counterScale, counterScale, point.x, point.y);

        // Draw handle shifted by half or width and height to match center of drawable and point coordinates
        canvas.drawBitmap(handle, point.x - (handle.getWidth() >> 1), point.y - (handle.getHeight() >> 1), handlePaint);

        canvas.restore();
    }

    /**
     * Keeps point inside of image area.
     * if coordinates of given point come out of image bounds
     * they get changed to nearest maximum/minimum values
     *
     * @param point
     */
    private void keepInsideView(PointF point) {
        float x = Math.max(0, Math.min(point.x, getWidth() - 2 * offsetX));
        float y = Math.max(0, Math.min(point.y, getHeight() - 2 * offsetY));

        point.set(x, y);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                int halfHandleSize = handleSize >> 1;
                touchRect.set(
                        event.getX() - offsetX - halfHandleSize,
                        event.getY() - offsetY - halfHandleSize,
                        event.getX() - offsetX + halfHandleSize,
                        event.getY() - offsetY + halfHandleSize);

                for (PointF point : corners) {
                    if (touchRect.contains(point.x, point.y)) {
                        selectedCorner = point;
                        invalidate();
                        return true;
                    }
                }

                for (Edge edge : edges) {
                    PointF edgeHandle = edge.getHandle();
                    if (touchRect.contains(edgeHandle.x, edgeHandle.y)) {
                        selectedEdge = edge;
                        edge.prevX = event.getX();
                        edge.prevY = event.getY();

                        edge.lineA.calculateLine(edge.pointA, edge.pointFarA);
                        edge.lineB.calculateLine(edge.pointB, edge.pointFarB);

                        finger.set(edgeHandle);

                        // TODO check for parallel lines and dismiss
                        invalidate();
                        return true;
                    }
                }

                return false;
            case MotionEvent.ACTION_MOVE:
                if (selectedCorner != null) {
                    selectedCorner.set(event.getX() - offsetX, event.getY() - offsetY);
                    keepInsideView(selectedCorner);
                    ensureNoPolygonIntersection();
                    invalidate();
                } else if (selectedEdge != null) {
                    float dx = event.getX() - selectedEdge.prevX;
                    float dy = event.getY() - selectedEdge.prevY;

                    finger.offset(dx, dy);

                    tmpPointA.set(selectedEdge.pointA);
                    tmpPointB.set(selectedEdge.pointB);

                    /**
                     * Find closest detected line
                     */
                    double closestDistance = Double.MAX_VALUE;
                    Line2D closestLine2D = null;

                    if (selectedEdge.isHorizontal()) {
                        for (int i = 0; i < horizontalLines.size(); i++) {
                            Line line = horizontalLines.get(i);
                            double distance = line.getDistanceToPoint(finger);
                            if (distance < closestDistance) {
                                closestDistance = distance;
                                closestLine2D = horizontalLines2D.get(i);
                            }
                        }
                    } else {
                        for (int i = 0; i < verticalLines.size(); i++) {
                            Line line = verticalLines.get(i);
                            double distance = line.getDistanceToPoint(finger);
                            if (distance < closestDistance) {
                                closestDistance = distance;
                                closestLine2D = verticalLines2D.get(i);
                            }
                        }
                    }

                    if (closestLine2D != null && closestDistance < magneticLineTreshold) {
                        selectedEdge.pointA.set(closestLine2D.getStart());
                        selectedEdge.pointB.set(closestLine2D.getEnd());
                    } else {
                        PointF handle = selectedEdge.getHandle();
                        float fingerDx = finger.x - handle.x;
                        float fingerDy = finger.y - handle.y;
                        selectedEdge.pointA.offset(fingerDx, fingerDy);
                        selectedEdge.pointB.offset(fingerDx, fingerDy);
                    }

                    /**
                     * calculating edge line to find new intersection points with line A and B
                     */
                    selectedEdge.edgeLine.calculateLine(selectedEdge.pointA, selectedEdge.pointB);

                    /**
                     * calculating intersection points and cancelling current edit in case if point
                     * coordinates are NaN
                     */
                    PointF intersectionA = selectedEdge.edgeLine.getIntersectionPoint(selectedEdge.lineA);
                    if (Float.isNaN(intersectionA.x) || Float.isNaN(intersectionA.y)) {
                        restoreSelectedEdge();
                        return false;
                    }
                    selectedEdge.pointA.set(intersectionA);

                    PointF intersectionB = selectedEdge.edgeLine.getIntersectionPoint(selectedEdge.lineB);
                    if (Float.isNaN(intersectionB.x) || Float.isNaN(intersectionB.y)) {
                        restoreSelectedEdge();
                        return false;
                    }
                    selectedEdge.pointB.set(intersectionB);

                    keepInsideView(selectedEdge.pointA);
                    keepInsideView(selectedEdge.pointB);

                    selectedEdge.prevX = event.getX();
                    selectedEdge.prevY = event.getY();
                    invalidate();
                }

                return true;
            case MotionEvent.ACTION_CANCEL:
            case MotionEvent.ACTION_UP:
                selectedCorner = null;
                selectedEdge = null;

                if (magnifierListener != null) {
                    magnifierListener.eraseMagnifier();
                }
                invalidate();
                return true;
        }

        return super.onTouchEvent(event);
    }



    private void restoreSelectedEdge() {
        selectedEdge.pointA.set(tmpPointA);
        selectedEdge.pointB.set(tmpPointB);
        selectedEdge.edgeLine.calculateLine(selectedEdge.pointA, selectedEdge.pointB);
    }

    private void ensureNoPolygonIntersection() {
        calculateMedian();

        if (ensureCornersOrder()) {
            ensureEdgesOrder();
        }
    }

    private boolean ensureCornersOrder() {
        boolean changed = false;

        pointsQueue.addAll(corners);
        for (int i = 0; i < CORNERS_COUNT; i++) {
            PointF pointF = pointsQueue.remove();

            if (!pointF.equals(corners.get(i))) {
                changed = true;
            }

            corners.set(i, pointF);
        }

        return changed;
    }

    private void calculateMedian() {
        float sx = 0f;
        float sy = 0f;

        for (int i = 0; i < CORNERS_COUNT; i++) {
            PointF pointF = corners.get(i);
            sx += pointF.x;
            sy += pointF.y;
        }

        medianX = sx / (float) CORNERS_COUNT;
        medianY = sy / (float) CORNERS_COUNT;
    }

    private void ensureEdgesOrder() {
        for (int i = 0; i < CORNERS_COUNT; i++) {
            Edge edge = edges.get(i);
            edge.pointA = corners.get(i);
            edge.pointB = corners.get((i + 1) % CORNERS_COUNT);
            edge.pointFarA = corners.get((i + 3) % CORNERS_COUNT);
            edge.pointFarB = corners.get((i + 2) % CORNERS_COUNT);
        }
    }

    private void rotateAndScale() {
        rotationScale = calculateRotationScale();
        setScaleX(rotationScale);
        setScaleY(rotationScale);

        float counterScale = 1 / rotationScale;
        paint.setStrokeWidth(counterScale * polygonStrokeWidth);
    }

    private float calculateRotationScale() {
        if (getRotation() != 90 && getRotation() != 270) {
            return ViewUtils.SCALE_DEFAULT;
        }

        final float availableWidth = getWidth();
        final float availableHeight = getHeight();

        return Math.min(
                availableWidth / (availableHeight - offsetY * 2),
                availableHeight / (availableWidth - offsetX * 2)
        );
    }

    /**
     * Represents edge of polygon with near and far points
     * as well as intersecting lines from other edges
     * <p/>
     *
     *    pointA     handle     pointB
     *       O========[0]=========O
     *       |   ^ edgeLine ^     |
     *       |                    |
     * lineA |                    | lineB
     *       |                    |
     *       |                    |
     *       O====================O
     *   pointFarA             pointFarB
     */
    private class Edge {
        private PointF pointA;
        private PointF pointB;
        private PointF pointFarA;
        private PointF pointFarB;

        private Line edgeLine = new Line();
        private Line lineA = new Line();
        private Line lineB = new Line();

        private PointF handle = new PointF();

        private float prevX;
        private float prevY;

        /**
         * @return edge handle rotation angle in degrees
         */
        float getAngleInDegrees() {
            float dy = pointB.y - pointA.y;
            float dx = pointB.x - pointA.x;
            double angle = Math.atan2(dy, dx);
            angle = Math.toDegrees(angle);

            // compensation for edge handle drawable
            // TODO remove if edge handle will be horizontal
            angle += 90;
            return (float) angle;
        }

        /**
         * @return handle for ths edge in the middle of segment AB
         */
        PointF getHandle() {
            handle.x = (pointA.x + pointB.x) / 2;
            handle.y = (pointA.y + pointB.y) / 2;
            return handle;
        }

        /**
         * @return {@code true} if edge is more horizontal than vertical, {@code false} otherwise
         */
        boolean isHorizontal() {
            float dy = pointB.y - pointA.y;
            float dx = pointB.x - pointA.x;
            return Math.abs(dx) > Math.abs(dy);
        }
    }

    /**
     * Representation of line
     * <p/>
     * Calculates line by two points
     * and finds intersection points with other lines
     * <p/>
     * General line formula is
     * a * x + b * y + c = 0
     */
    private class Line {
        private double a;
        private double b;
        private double c;

        /**
         * One for line instance to reduce GC calls
         */
        private PointF intersectionPoint = new PointF();

        /**
         * @param line
         * @return intersection point between current and given lines
         */
        PointF getIntersectionPoint(Line line) {
            intersectionPoint.x = (float) ((c * line.b - b * line.c) / (b * line.a - a * line.b));
            intersectionPoint.y = (float) ((a * line.c - c * line.a) / (b * line.a - a * line.b));

            return intersectionPoint;
        }

        /**
         * Calculate all 3 components of the line
         *
         * @param start starting point of segment laying on this line
         * @param end   ending point of segment laying on this line
         */
        void calculateLine(PointF start, PointF end) {
            a = start.y - end.y;
            b = end.x - start.x;
            c = (start.x - end.x) * start.y + (end.y - start.y) * start.x;
        }

        /**
         * Calculates distance to point
         * Should be used after line components are calculated
         *
         * @param point
         * @return
         */
        double getDistanceToPoint(PointF point) {
            if (a != 0 || b != 0) {
                return Math.abs(a * point.x + b * point.y + c) / Math.sqrt(a * a + b * b);
            }

            return 0;
        }
    }
}
