package net.doo.snap.lib.persistence;

import android.app.Application;
import android.os.AsyncTask;
import android.os.Looper;

import com.google.inject.Inject;

import net.doo.snap.lib.util.log.DebugLog;

import org.apache.commons.io.FileUtils;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;

import static net.doo.snap.lib.util.FileUtils.getExternalFilesDirOrShowError;

/**
 * Provides {@link java.io.File} directories for saving {@link net.doo.snap.lib.persistence.Page}s
 *
 * @see net.doo.snap.lib.persistence.PageFactory
 */
public class PageStoreStrategy {

    public static final String FILTERED_FOLDER = "filtered";

    private Application context;

    @Inject
    public PageStoreStrategy(Application context) {
        this.context = context;
    }

    /**
     * @return {@link File} directory where {@link net.doo.snap.lib.persistence.Page}s will be stored or {@code null} if not possible to create
     * @throws java.io.IOException if directory can't be opened
     */
    @NotNull
    public File getPagesDir() throws IOException {
        return getExternalFilesDirOrShowError(context, "snapping_pages");
    }

    /**
     * @return {@link java.io.File} directory where {@link net.doo.snap.lib.persistence.Page} stored
     * or {@code null} if thehre wre problems accessing FS
     * @throws java.io.IOException if directory can't be opened
     */
    @NotNull
    public File getPageDir(String pageId) throws IOException {
        return FileUtils.getFile(getPagesDir(), pageId);
    }

    /**
     * @return image {@link java.io.File} of provided {@link net.doo.snap.lib.persistence.Page.ImageType} for this {@link net.doo.snap.lib.persistence.Page}
     * @throws IOException if {@link java.io.File} for {@link net.doo.snap.lib.persistence.Page} can't be opened
     */
    @NotNull
    public File getImageFile(@NotNull String pageId, @NotNull Page.ImageType imageType) throws IOException {
        return FileUtils.getFile(
                getPagesDir(),
                pageId,
                imageType.getFileName()
        );
    }

    /**
     * @param optimizationType {@link net.doo.snap.lib.persistence.OptimizationType} used for filtration of image
     * @return file name of filtered image
     * @throws java.io.IOException if {@link java.io.File} for {@link net.doo.snap.lib.persistence.Page} can't be opened
     */
    public File getFilteredPreviewFile(@NotNull String pageId, @NotNull OptimizationType optimizationType) throws IOException {
        return FileUtils.getFile(
                getPageDir(pageId),
                FILTERED_FOLDER,
                optimizationType.getName()
        );
    }

    /**
     * Erase page directory from filesystem
     *
     * @param pageId
     */
    public void erasePage(final String pageId) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            new AsyncTask<String, Void, Void>() {

                @Override
                protected Void doInBackground(String... params) {
                    String pageId = params[0];
                    erasePageSync(pageId);
                    return null;
                }
            }.execute(pageId);
        } else {
            erasePageSync(pageId);
        }
    }

    /**
     * Erase page directory synchronously
     *
     * @param pageId
     */
    private void erasePageSync(String pageId) {
        try {
            FileUtils.deleteDirectory(getPageDir(pageId));
        } catch (IOException e) {
            DebugLog.logException(e);
        }
    }
}
