package net.doo.snap.lib.snap.camera.barcode.result;

import android.app.Activity;
import android.telephony.PhoneNumberUtils;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.TextUtils;
import android.text.style.StyleSpan;

import com.google.zxing.client.result.AddressBookParsedResult;
import com.google.zxing.client.result.ParsedResult;

import net.doo.snap.lib.R;
import net.doo.snap.lib.util.ThemesHelper;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * Handles address book entries.
 */
public final class AddressBookResultHandler extends ResultHandler {

    private static final DateFormat[] DATE_FORMATS = {
            new SimpleDateFormat("yyyyMMdd", Locale.ENGLISH),
            new SimpleDateFormat("yyyyMMdd'T'HHmmss", Locale.ENGLISH),
            new SimpleDateFormat("yyyy-MM-dd", Locale.ENGLISH),
            new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.ENGLISH),
    };

    static {
        for (DateFormat format : DATE_FORMATS) {
            format.setLenient(false);
        }
    }

    private static final int[] BUTTON_TEXTS = {
            R.string.button_add_contact,
            R.string.button_show_map,
            R.string.button_dial,
            R.string.button_email,
    };

    private static final int DISABLED_FIELD = -1;
    private static final int ADD_TO_CONTACTS = 0;
    private static final int SEARCH_MAP = 1;
    private static final int CALL = 2;
    private static final int SEND_MAIL = 3;

    private final List<Integer> fields;
    private int buttonCount;

    public AddressBookResultHandler(Activity activity, AddressBookParsedResult addressResult) {
        super(activity, addressResult);
        String[] addresses = getResult().getAddresses();
        boolean hasAddress = addresses != null && addresses.length > 0 && addresses[0] != null && !addresses[0].isEmpty();
        String[] phoneNumbers = getResult().getPhoneNumbers();
        boolean hasPhoneNumber = phoneNumbers != null && phoneNumbers.length > 0;
        String[] emails = getResult().getEmails();
        boolean hasEmailAddress = emails != null && emails.length > 0;

        int index = 0;
        fields = Arrays.asList(
                index++, // Add contact is always available
                hasAddress ? index++ : DISABLED_FIELD,
                hasPhoneNumber ? index++ : DISABLED_FIELD,
                hasEmailAddress ? index++ : DISABLED_FIELD
        );

        buttonCount = index;
    }

    // This takes all the work out of figuring out which buttons/actions should be in which
    // positions, based on which fields are present in this barcode.
    private int mapIndexToAction(int index) {
        return fields.indexOf(index);
    }

    @Override
    public AddressBookParsedResult getResult() {
        return (AddressBookParsedResult) super.getResult();
    }

    @Override
    public int getButtonCount() {
        return buttonCount;
    }

    @Override
    public int getButtonText(int index) {
        return BUTTON_TEXTS[mapIndexToAction(index)];
    }

    @Override
    public void handleButtonPress(int index) {
        AddressBookParsedResult addressResult = getResult();
        String[] addresses = addressResult.getAddresses();
        String address1 = addresses == null || addresses.length < 1 ? null : addresses[0];
        String[] addressTypes = addressResult.getAddressTypes();
        String address1Type = addressTypes == null || addressTypes.length < 1 ? null : addressTypes[0];
        int action = mapIndexToAction(index);
        switch (action) {
            case ADD_TO_CONTACTS:
                addContact(addressResult.getNames(),
                        addressResult.getNicknames(),
                        addressResult.getPronunciation(),
                        addressResult.getPhoneNumbers(),
                        addressResult.getPhoneTypes(),
                        addressResult.getEmails(),
                        addressResult.getEmailTypes(),
                        addressResult.getNote(),
                        addressResult.getInstantMessenger(),
                        address1,
                        address1Type,
                        addressResult.getOrg(),
                        addressResult.getTitle(),
                        addressResult.getURLs(),
                        addressResult.getBirthday(),
                        addressResult.getGeo());
                break;
            case SEARCH_MAP:
                searchMap(address1);
                break;
            case CALL:
                dialPhone(addressResult.getPhoneNumbers()[0]);
                break;
            case SEND_MAIL:
                sendEmail(addressResult.getEmails()[0], null, null);
                break;
        }
    }

    private static Date parseDate(String s) {
        for (DateFormat currentFormat : DATE_FORMATS) {
            try {
                return currentFormat.parse(s);
            } catch (ParseException e) {
                // continue
            }
        }
        return null;
    }

    // Overriden so we can hyphenate phone numbers, format birthdays, and bold the name.
    @Override
    public CharSequence getDisplayContents() {
        AddressBookParsedResult result = getResult();
        StringBuilder contents = new StringBuilder(100);
        ParsedResult.maybeAppend(result.getNames(), contents);
        int namesLength = contents.length();

        String pronunciation = result.getPronunciation();
        if (!TextUtils.isEmpty(pronunciation)) {
            contents.append("\n(");
            contents.append(pronunciation);
            contents.append(')');
        }

        ParsedResult.maybeAppend(result.getTitle(), contents);
        ParsedResult.maybeAppend(result.getOrg(), contents);
        ParsedResult.maybeAppend(result.getAddresses(), contents);
        String[] numbers = result.getPhoneNumbers();
        if (numbers != null) {
            for (String number : numbers) {
                if (!TextUtils.isEmpty(number)) {
                    ParsedResult.maybeAppend(PhoneNumberUtils.formatNumber(number), contents);
                }
            }
        }
        ParsedResult.maybeAppend(result.getEmails(), contents);
        ParsedResult.maybeAppend(result.getURLs(), contents);

        String birthday = result.getBirthday();
        if (!TextUtils.isEmpty(birthday)) {
            Date date = parseDate(birthday);
            if (date != null) {
                ParsedResult.maybeAppend(DateFormat.getDateInstance(DateFormat.MEDIUM).format(date.getTime()), contents);
            }
        }
        ParsedResult.maybeAppend(result.getNote(), contents);

        if (namesLength > 0) {
            // Bold the full name to make it stand out a bit.
            Spannable styled = new SpannableString(contents.toString());
            styled.setSpan(new StyleSpan(android.graphics.Typeface.BOLD), 0, namesLength, 0);
            return styled;
        } else {
            return contents.toString();
        }
    }

    @Override
    public String getDisplayTitle() {
        AddressBookParsedResult addressResult = getResult();
        return (addressResult.getNames().length == 0) ? "" : TextUtils.join(" ", addressResult.getNames());
    }

    @Override
    public String getDisplaySubtitle() {
        return getResult().getOrg();
    }

    @Override
    public String getDisplayText() {
        return null;
    }

    @Override
    public int getDisplayIconRes() {
        return ThemesHelper.getResourceId(getActivity(), R.attr.ui_qr_ico_contact);
    }
}
