package net.doo.snap.lib.snap;

import android.content.Context;
import android.content.res.Configuration;
import android.view.Gravity;
import android.view.View;
import android.widget.ImageView;
import android.widget.Toast;

import net.doo.snap.lib.R;
import net.doo.snap.lib.detector.DetectionResult;
import net.doo.snap.lib.util.ThemesHelper;

/**
 * Helper to show visual hints while auto snapping is active
 */
public class AutoSnappingDetectionHelper {
    private final Context context;
    private Toast toast;
    private ImageView rotateDevice;
    private View moveCloser;
    private int orientation;

    /**
     *
     * @param context
     * @param root View which contains helper views to show them to user,
     *             such as R.id.rotate_device and R.id.move_closer
     */
    public AutoSnappingDetectionHelper(Context context, View root) {
        this.context = context;
        toast = Toast.makeText(context, "", Toast.LENGTH_SHORT);
        toast.setGravity(Gravity.CENTER, 0, 0);
        orientation = context.getResources().getConfiguration().orientation;

        rotateDevice = (ImageView) root.findViewById(R.id.rotate_device);
        moveCloser = root.findViewById(R.id.move_closer);
    }

    /**
     * Hides all visual hints
     */
    public void reset(boolean cancelToast) {
        rotateDevice.setVisibility(View.GONE);
        moveCloser.setVisibility(View.GONE);

        if (cancelToast) {
            toast.cancel();
        }
    }

    /**
     * Shows hints to help user to use auto snapping
     * @param result detection result from {@link net.doo.snap.lib.detector.ContourDetector}
     */
    public void onResult(DetectionResult result) {
        reset(false);
        switch (result) {
            case OK_BUT_BAD_ASPECT_RATIO:
                toast.cancel();
                int imageResource = ThemesHelper.getResourceId(context, R.attr.ui_cam_rotation_h);
                switch (orientation) {
                    case Configuration.ORIENTATION_LANDSCAPE:
                        imageResource = ThemesHelper.getResourceId(context, R.attr.ui_cam_rotation_v);
                        break;
                }
                rotateDevice.setImageResource(imageResource);
                rotateDevice.setVisibility(View.VISIBLE);
                break;
            case OK_BUT_TOO_SMALL:
                moveCloser.setVisibility(View.VISIBLE);
                toast.setText(R.string.autosnapping_hint_move_closer);
                toast.show();
                break;
            case OK_BUT_BAD_ANGLES:
                toast.setText(R.string.autosnapping_hint_bad_angles);
                toast.show();
                break;
            case ERROR_NOTHING_DETECTED:
                toast.setText(R.string.autosnapping_hint_nothing_detected);
                toast.show();
                break;
            case ERROR_TOO_NOISY:
                toast.setText(R.string.autosnapping_hint_too_noisy);
                toast.show();
                break;
            default:
                toast.cancel();
                break;
        }
    }
}
