package net.doo.snap.lib.snap.camera.barcode;

import com.google.inject.Inject;
import com.google.zxing.*;
import com.google.zxing.common.HybridBinarizer;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Manages barcode detection operations and detector initialization
 */
public class BarcodeDetector {
    private static final int BARCODE_SCANNING_INTERVAL_SEC = 7;

    private final MultiFormatReader multiFormatReader;
    private Result lastResult;

    private AtomicBoolean barcodeScanEnabled = new AtomicBoolean(true);
    private ScheduledExecutorService scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
    private ScheduledFuture<?> schedule;

    @Inject
    public BarcodeDetector() {
        this.multiFormatReader = new MultiFormatReader();
    }

    /**
     * @param barcodeScanEnabled if {@code true} barcode scanning will be enabled, {@code false} - will be disabled
     */
    public void enableBarcodeScan(boolean barcodeScanEnabled) {
        if (schedule != null && !schedule.isDone()) {
            if (!barcodeScanEnabled) {
                schedule.cancel(false);
            }
        } else {
            this.barcodeScanEnabled.set(barcodeScanEnabled);
        }
    }

    /**
     * @return last detected barcode content
     */
    public Result getLastResult() {
        return lastResult;
    }

    /**
     * Decode the data within the viewfinder rectangle, and time how long it took. For efficiency,
     * reuse the same reader objects from one decode to the next.
     *
     * @param data   The YUV preview frame.
     * @param width  The width of the preview frame.
     * @param height The height of the preview frame.
     */
    public Result decodeWithState(byte[] data, int width, int height) {
        if (barcodeScanEnabled.compareAndSet(false, false)) {
            return null;
        }
        try {
            PlanarYUVLuminanceSource source = new PlanarYUVLuminanceSource(data, width, height, 0, 0,
                    width, height, false);
            BinaryBitmap bitmap = new BinaryBitmap(new HybridBinarizer(source));
            lastResult = multiFormatReader.decodeWithState(bitmap);
            if (lastResult != null) {
                barcodeScanEnabled.set(false);
            }
            return lastResult;
        } catch (ReaderException re) {
            // continue
        } finally {
            multiFormatReader.reset();
        }
        return null;
    }

    /**
     * Starts #BARCODE_SCANNING_INTERVAL_SEC seconds timer countdown
     */
    public void resetBarcodeDetectionTimer() {
        schedule = scheduledExecutor.schedule(new Runnable() {
            @Override
            public void run() {
                barcodeScanEnabled.set(true);
            }
        }, BARCODE_SCANNING_INTERVAL_SEC, TimeUnit.SECONDS);
    }
}
