package net.doo.snap.lib.ui.widget.text.processor;

import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.drawable.Drawable;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.style.DynamicDrawableSpan;

/**
 * Formats input with custom bullets.
 * <p/>
 * Format rules:
 * - Double-dash "--" will be replaced with a {@link android.graphics.drawable.Drawable}. If it's not a
 * first character in a text, double-dash is prepended with '\n'
 */
public class BulletProcessStrategy implements TextProcessor.ProcessStrategy {

    private static final char DASH_CHARACTER = '-';

    private static final int STATE_DEFAULT = 0;
    private static final int STATE_FIRST_DASH = 1;

    private final Drawable bullet;

    private int state = STATE_DEFAULT;

    public BulletProcessStrategy(Drawable bullet) {
        this.bullet = bullet;
    }

    @Override
    public boolean processChar(char character, SpannableStringBuilder output) {
        if (character != DASH_CHARACTER) {
            if (state == STATE_FIRST_DASH) {
                output.append(DASH_CHARACTER);
            }

            state = STATE_DEFAULT;
            return false;
        }

        if (state == STATE_DEFAULT) {
            state = STATE_FIRST_DASH;
            return true;
        }

        if (state == STATE_FIRST_DASH) {
            if (output.length() > 0) {
                output.append('\n');
            }

            output.append(' ');

            output.setSpan(
                    new BulletSpan(bullet),
                    output.length() - 1,
                    output.length(),
                    Spanned.SPAN_EXCLUSIVE_EXCLUSIVE
            );

            output.append(' ');

            state = STATE_DEFAULT;
            return true;
        }

        return false;
    }

    @Override
    public void reset() {
        state = STATE_DEFAULT;
    }

    /**
     * Span that displays drawable, which will be vertically centered in line
     */
    private static class BulletSpan extends DynamicDrawableSpan {

        private final Drawable drawable;

        private BulletSpan(Drawable drawable) {
            super(ALIGN_BASELINE);
            this.drawable = drawable;
        }

        @Override
        public Drawable getDrawable() {
            drawable.setBounds(
                    0, 0,
                    drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight()
            );

            return drawable;
        }

        @Override
        public void draw(Canvas canvas, CharSequence text, int start, int end, float x, int top, int y, int bottom, Paint paint) {
            canvas.save();
            canvas.translate(x, y - drawable.getIntrinsicHeight() + (paint.descent() - paint.getTextSize() + drawable.getIntrinsicHeight()) / 3f);
            drawable.draw(canvas);
            canvas.restore();
        }
    }

}
