package net.doo.snap.lib.persistence;

import android.content.SharedPreferences;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.Point;
import android.graphics.PointF;
import android.util.DisplayMetrics;

import com.google.inject.Inject;

import net.doo.snap.lib.Constants;
import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.detector.ContourDetector;
import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.log.DebugLog;
import net.doo.snap.lib.util.snap.PolygonHelper;

import org.apache.commons.io.IOUtils;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Processes raw input (like bytes) using native libraries, saves result and generates new {@link net.doo.snap.lib.persistence.Page}
 */
public class PictureProcessor {

    private static final ExecutorService executor = Executors.newSingleThreadExecutor();
    private static final float FILTERED_PREVIEW_SCALE = 0.65f;

    @Inject
    private BitmapLruCache bitmapLruCache;

    private PageFactory pageFactory;
    private PageStoreStrategy pageStoreStrategy;
    private Resources resources;
    private SharedPreferences preferences;

    @Inject
    public PictureProcessor(PageFactory pageFactory, PageStoreStrategy pageStoreStrategy, Resources resources, SharedPreferences preferences) {
        this.pageFactory = pageFactory;
        this.pageStoreStrategy = pageStoreStrategy;
        this.resources = resources;
        this.preferences = preferences;
    }

    /**
     * Processes byte array using native libraries, saves result and generates new {@link net.doo.snap.lib.persistence.Page}
     *
     * @param image            image in {@link android.graphics.ImageFormat#JPEG} format
     * @param imageOrientation image orientation in degrees
     * @return {@link net.doo.snap.lib.persistence.Page} which is associated with processed image
     * @throws IOException if image cannot be saved
     */
    public Page processBytes(byte[] image, int imageOrientation) throws IOException {
        DisplayMetrics displayMetrics = resources.getDisplayMetrics();

        final PageFactory.Result result = pageFactory.buildPage(image, displayMetrics.widthPixels, displayMetrics.heightPixels);
        final Page page = result.page;

        page.setRotationType(RotationType.getByDegrees(imageOrientation));
        if (preferences.contains(PreferencesConstants.LAST_USED_FILTER)) {
            page.setOptimizationType(OptimizationType.getByCode(preferences.getInt(PreferencesConstants.LAST_USED_FILTER, OptimizationType.NONE.getCode())));
        }

        ContourDetector detector = new ContourDetector();
        detector.detect(result.preview);
        List<PointF> polygonF = detector.getPolygonF();
        List<Point> polygon = detector.getPolygon();
        if (!polygon.isEmpty() && PolygonHelper.checkPolygonSize(polygon)) {
            page.setPolygon(polygonF);
        }
        generateOptimizedPreview(pageStoreStrategy, page, result.preview, detector);

        return page;
    }

    private void generateOptimizedPreview(PageStoreStrategy pageStoreStrategy, Page page, Bitmap preview, ContourDetector detector) throws IOException {
        List<PointF> polygon = page.getPolygon();

        if (!polygon.isEmpty()) {
            final Bitmap bitmap = detector.processImageF(preview, polygon, page.getOptimizationType().getCode());

            final String path = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.OPTIMIZED_PREVIEW).getPath();

            bitmapLruCache.put(path, bitmap);

            executor.execute(new Runnable() {
                @Override
                public void run() {
                    FileOutputStream fos = null;
                    try {
                        fos = new FileOutputStream(path);
                        bitmap.compress(Bitmap.CompressFormat.JPEG, Constants.JPEG_QUALITY, fos);
                    } catch (IOException e) {
                        DebugLog.logException(e);
                    } finally {
                        IOUtils.closeQuietly(fos);
                    }
                }
            });

            generateFilteredPreviewAsync(pageStoreStrategy, page, preview, detector);
        }
    }

    /**
     * Generates optimized and cropped preview
     * using {@link net.doo.snap.lib.persistence.Page.ImageType#PREVIEW} as a source
     *
     * @param page
     * @param detector
     * @throws IOException when file not found or it is not possible to close FileOutputStream
     */
    public static void generateOptimizedPreview(PageStoreStrategy pageStoreStrategy, Page page, ContourDetector detector) throws IOException {
        List<PointF> polygon = page.getPolygon();

        if (!polygon.isEmpty()) {
            final String previewPath = pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.PREVIEW).getPath();
            final Bitmap preview = BitmapFactory.decodeFile(previewPath);
            Bitmap bitmap = detector.processImageF(preview, polygon, page.getOptimizationType().getCode());

            FileOutputStream fos = null;
            try {
                fos = new FileOutputStream(pageStoreStrategy.getImageFile(page.getId(), Page.ImageType.OPTIMIZED_PREVIEW));
                bitmap.compress(Bitmap.CompressFormat.JPEG, Constants.JPEG_QUALITY, fos);
            } finally {
                IOUtils.closeQuietly(fos);
            }

            generateFilteredPreviewAsync(pageStoreStrategy, page, preview, detector);
        }
    }

    private static void generateFilteredPreviewAsync(final PageStoreStrategy pageStoreStrategy, final Page page, final Bitmap preview, final ContourDetector detector) {
        executor.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    generateFilteredPreview(pageStoreStrategy, page, preview, detector);
                } catch (IOException e) {
                    DebugLog.logException(e);
                }
            }
        });
    }

    private static void generateFilteredPreview(PageStoreStrategy pageStoreStrategy, Page page, Bitmap preview, ContourDetector detector) throws IOException {
        Matrix matrix = new Matrix();
        matrix.setScale(FILTERED_PREVIEW_SCALE, FILTERED_PREVIEW_SCALE);

        Bitmap downscaledPreview = Bitmap.createBitmap(preview, 0, 0, preview.getWidth(), preview.getHeight(), matrix, true);
        for (OptimizationType optimizationType : OptimizationType.values()) {
            Bitmap processed = detector.processImageF(
                    downscaledPreview,
                    page.getPolygon(),
                    optimizationType.getCode()
            );

            FileOutputStream fos = new FileOutputStream(
                    pageStoreStrategy.getFilteredPreviewFile(page.getId(), optimizationType)
            );
            try {
                processed.compress(
                        Bitmap.CompressFormat.JPEG,
                        Constants.JPEG_QUALITY,
                        fos
                );
            } finally {
                IOUtils.closeQuietly(fos);
            }
        }
    }

}
