package net.doo.snap.lib.util.snap;

import android.hardware.Camera;
import android.os.Build;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import crl.android.pdfwriter.PaperSize;

/**
 * Helper methods to work with {@link android.hardware.Camera}
 */
public class Utils {
    private static final int A4_AREA = PaperSize.A4_PRINT_WIDTH * PaperSize.A4_PRINT_HEIGHT;

    private static final double ASPECT_TOLERANCE = 0.1;

    /**
     * @param parameters Currently selected camera parameters
     * @return true if flash is supported for this camera, false otherwise
     */
    public static boolean isFlashSupported(Camera.Parameters parameters) {
        if (parameters == null) {
            return false;
        }

        List<String> supportedFlashModes = parameters.getSupportedFlashModes();
        if (supportedFlashModes == null
                || supportedFlashModes.isEmpty()
                || (supportedFlashModes.size() == 1 && supportedFlashModes.get(0).equals(Camera.Parameters.FLASH_MODE_OFF))) {
            return false;
        }

        return true;
    }

    /**
     * @param parameters
     * @return Best picture size that is suitable to print on A4
     * <p/>
     * 210mm X 297mm @ 300 dpi is about 2480 X 3508 pixels
     */
    public static Camera.Size getBestA4PictureSize(Camera.Parameters parameters) {
        TreeMap<Integer, Camera.Size> sizes = new TreeMap<Integer, Camera.Size>();

        for (Camera.Size size : parameters.getSupportedPictureSizes()) {
            int area = size.width * size.height;
            sizes.put(area, size);
        }

        Camera.Size result = null;

        Map.Entry<Integer, Camera.Size> entry = sizes.ceilingEntry(A4_AREA);
        if (entry == null) {
            entry = sizes.floorEntry(A4_AREA);
        }

        if (entry != null) {
            result = entry.getValue();
        }

        return result;
    }

    /**
     * Returns the biggest size for resulting image
     */
    public static Camera.Size getLargestPictureSize(Camera.Parameters parameters) {
        Camera.Size result = null;

        for (Camera.Size size : parameters.getSupportedPictureSizes()) {
            if (result == null) {
                result = size;
            } else {
                int resultArea = result.width * result.height;
                int newArea = size.width * size.height;

                if (newArea > resultArea) {
                    result = size;
                }
            }
        }

        return result;
    }

    /**
     * @return largest {@link android.hardware.Camera.Size} from list with given aspect ratio.
     */
    public static Camera.Size getLargestSizeWithAspectRatioMatch(List<Camera.Size> sizes, double targetAspect) {
        Comparator<Camera.Size> comparator = new SizeComparator();

        ArrayList<Camera.Size> matchedPictureSizes = new ArrayList<Camera.Size>();
        for (Camera.Size size : sizes) {
            final double ratio = size.width / (double) size.height;

            if (Math.abs(targetAspect - ratio) < ASPECT_TOLERANCE) {
                matchedPictureSizes.add(size);
            }
        }

        return Collections.max(matchedPictureSizes, comparator);
    }

    /**
     * @return {@link android.hardware.Camera.Size} with given width and height, or {@code null} if there
     * is no such supported size.
     */
    public static Camera.Size findByWidthAndHeight(Camera.Parameters parameters, int width, int height) {
        for (Camera.Size size : parameters.getSupportedPictureSizes()) {
            if (size.width == width && size.height == height) {
                return size;
            }
        }

        return null;
    }

    /**
     * @param displayOrientation display orientation angle (0-270)
     * @param width              target display port width
     * @param height             target display port height
     * @param parameters         {@link android.hardware.Camera.Parameters} to get supported sizes from
     * @param closeEnough        maximum deviation to stop searching for best size, pass 0 to go through all sizes
     * @return {@link android.hardware.Camera.Size} that matches display port most
     */
    public static Camera.Size getBestAspectPictureSize(int displayOrientation,
                                                       int width,
                                                       int height,
                                                       Camera.Parameters parameters,
                                                       double closeEnough) {
        double targetRatio = (double) width / height;
        Camera.Size optimalSize = null;
        double minDiff = Double.MAX_VALUE;

        if (displayOrientation == 90 || displayOrientation == 270) {
            targetRatio = (double) height / width;
        }

        List<Camera.Size> sizes = parameters.getSupportedPictureSizes();

        Collections.sort(sizes,
                Collections.reverseOrder(new SizeComparator()));

        for (Camera.Size size : sizes) {
            double ratio = (double) size.width / size.height;

            if (Math.abs(ratio - targetRatio) < minDiff) {
                optimalSize = size;
                minDiff = Math.abs(ratio - targetRatio);
            }

            if (minDiff < closeEnough) {
                break;
            }
        }

        return optimalSize;
    }

    /**
     * Sets focus mode to {@link Camera.Parameters#FOCUS_MODE_CONTINUOUS_PICTURE}
     * or {@link Camera.Parameters#FOCUS_MODE_AUTO} if supported
     *
     * @param parameters target parameters to set focus mode to
     */
    public static void enableContinuousFocus(Camera.Parameters parameters) {
        if (parameters == null) {
            return;
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH) {
            List<String> focusModes = extractFocusModes(parameters);
            if (focusModes.contains(Camera.Parameters.FOCUS_MODE_CONTINUOUS_PICTURE)) {
                parameters.setFocusMode(Camera.Parameters.FOCUS_MODE_CONTINUOUS_PICTURE);
            } else if (focusModes.contains(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO)) {
                parameters.setFocusMode(Camera.Parameters.FOCUS_MODE_CONTINUOUS_VIDEO);
            } else if (focusModes.contains(Camera.Parameters.FOCUS_MODE_AUTO)) {
                parameters.setFocusMode(Camera.Parameters.FOCUS_MODE_AUTO);
            }
        }
    }

    /**
     * Sets focus mode to {@link Camera.Parameters#FOCUS_MODE_AUTO} if supported
     *
     * @param parameters
     */
    public static void enableAutoFocus(Camera.Parameters parameters) {
        if (parameters == null) {
            return;
        }

        List<String> focusModes = extractFocusModes(parameters);
        if (focusModes.contains(Camera.Parameters.FOCUS_MODE_AUTO) && !parameters.getFocusMode().equals(Camera.Parameters.FOCUS_MODE_AUTO)) {
            parameters.setFocusMode(Camera.Parameters.FOCUS_MODE_AUTO);
        }
    }

    private static List<String> extractFocusModes(Camera.Parameters parameters) {
        if (parameters == null) {
            return Collections.<String>emptyList();
        }

        List<String> modes = parameters.getSupportedFocusModes();
        return (modes != null)
                ? modes
                : Collections.<String>emptyList();
    }

    /**
     * Compares {@link android.hardware.Camera.Size} according to total area (width * height).
     */
    private static class SizeComparator implements
            Comparator<Camera.Size> {
        @Override
        public int compare(Camera.Size lhs, Camera.Size rhs) {
            int left = lhs.width * lhs.height;
            int right = rhs.width * rhs.height;

            return left - right;
        }
    }
}
