package net.doo.snap.lib.genuineness;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.widget.Toast;

import com.google.inject.Inject;

import net.doo.snap.lib.BuildConfig;
import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.AnalyticsConst;
import net.doo.snap.lib.analytics.EasyTrackerWrapper;
import net.doo.snap.lib.persistence.DocumentDraft;
import net.doo.snap.lib.snap.event.CheckGenuinenessEvent;
import net.doo.snap.lib.snap.event.SaveDocumentEvent;
import net.doo.snap.lib.util.MarketPageOpener;
import net.doo.snap.lib.util.log.DebugLog;

import roboguice.event.EventManager;
import roboguice.event.Observes;

/**
 * Checks if app is Genuine or not
 */
public class GenuinenessChecker {
    private static final int DEBUG_SIGNATURE = 1105598010;
    private static final int RELEASE_SIGNATURE = 2027059470;
    private static final int AMAZON_SIGNATURE = -20283086;
    public static final String BASE64_PUBLIC_KEY = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAu82nw1cLa7TrVGRF00wnHci9NqHuVAnH5EFeyoUKhLBX9duv6guVRktGx5ad7K35SPy8ZksaIvZnL6j2IbUtjweRPP+tXewONGhV4j+7wROQ2G4k2zbBVo7bKXzgKEfbpbKTbH4iub0xdBxsvW6X4OwbZ1zDrC+b4pHlddJpP167XouU7DCuaErLnnchf/ZA/2yha9s4boqIYty6e9hrOwjDf13Co05k+FWHMI5hQBCwL/4oljB/GXHs0mDvUsLa9vos3jjNxGht3aheZBlgLRyIht8u/hN1ZWs4RyUqI9Mmok3CSajbcnQj8PLzuY274iqklg4shgAzoVTOv3it4wIDAQAB";

    private static final int ERROR_DEBUG_SIGNATURE = 2146616982;
    private static final int ERROR_RELEASE_SIGNATURE = 349857392;
    private static final int ERROR_DEBUGGABLE = 645892742;
    private static final int GOOD_LICENSE = -309808353;

    private Genuineness genuineness = Genuineness.NO;

    @Inject
    private Context context;
    @Inject
    private EventManager eventManager;
    @Inject
    private MarketPageOpener marketPageOpener;

    /**
     * 3 states of app Genuineness.
     * YES and UNKNOWN allow user to save documents
     * NO disallow to save documents
     */
    public enum Genuineness {
        YES, NO, UNKNOWN;
    }

    @Inject
    public GenuinenessChecker(Context context) {
        this.context = context;
        int signature = 0;

        try {
            Signature[] signatures = context.getPackageManager().getPackageInfo(context.getPackageName(), PackageManager.GET_SIGNATURES).signatures;
            signature = signatures[0].hashCode();
        } catch (PackageManager.NameNotFoundException e) {
            DebugLog.logException(e);
        }

        if (BuildConfig.DEBUG) {
            if (DEBUG_SIGNATURE == signature) {
                genuineness = Genuineness.UNKNOWN;
                return;
            } else {
                trackLicense(ERROR_DEBUG_SIGNATURE);
                genuineness = Genuineness.NO;
                return;
            }
        } else {
            if (RELEASE_SIGNATURE != signature && AMAZON_SIGNATURE != signature) {
                trackLicense(ERROR_RELEASE_SIGNATURE);
                genuineness = Genuineness.NO;
                return;
            }
        }

        boolean isDebuggable = (0 != (context.getApplicationInfo().flags &= ApplicationInfo.FLAG_DEBUGGABLE));
        if (isDebuggable) {
            trackLicense(ERROR_DEBUGGABLE);
            genuineness = Genuineness.NO;
            return;
        }

        genuineness = Genuineness.YES;
        trackLicense(GOOD_LICENSE);
    }

    /**
     * @return true if app is Genuine or false if not
     */
    public Genuineness getAppGenuineness() {
        return genuineness;
    }

    private void trackLicense(int reason) {

        switch (reason) {
            case GOOD_LICENSE:
                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_LICENSE, AnalyticsConst.ACTION_GENUINE, "license_check", 0L);
                break;
            case ERROR_DEBUG_SIGNATURE:
                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_LICENSE, AnalyticsConst.ACTION_NOT_GENUINE, "debug_signature", 0L);
                break;
            case ERROR_RELEASE_SIGNATURE:
                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_LICENSE, AnalyticsConst.ACTION_NOT_GENUINE, "release_signature", 0L);
                break;
            case ERROR_DEBUGGABLE:
                EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_LICENSE, AnalyticsConst.ACTION_NOT_GENUINE, "debuggable", 0L);
                break;
        }
    }

    @SuppressWarnings("unused")
    private void checkDocument(@Observes CheckGenuinenessEvent event) {
        DocumentDraft documentDraft = event.getDraft();

        if (getAppGenuineness().equals(Genuineness.NO)) {
            redirectToGooglePlay();
            EasyTrackerWrapper.sendEvent(AnalyticsConst.CATEGORY_LICENSE, AnalyticsConst.ACTION_UNAUTH_USE, "save_document", 0L);
            return;
        }

        eventManager.fire(new SaveDocumentEvent(documentDraft));
    }

    private void redirectToGooglePlay() {
        Toast.makeText(context, R.string.please_buy_app, Toast.LENGTH_LONG).show();

        marketPageOpener.openMarketPage();
    }
}
