package net.doo.snap.lib.snap.camera.barcode.result;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.text.TextUtils;

import com.google.zxing.client.result.CalendarParsedResult;
import com.google.zxing.client.result.ParsedResult;

import net.doo.snap.lib.R;
import net.doo.snap.lib.util.ThemesHelper;
import net.doo.snap.lib.util.log.DebugLog;

import org.apache.commons.lang.time.DateUtils;

import java.text.DateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Handles calendar entries encoded in QR Codes.
 */
public final class CalendarResultHandler extends ResultHandler {

    private static final int[] buttons = {
            R.string.button_add_calendar
    };

    private static final int ADD_TO_CALENDAR = 0;
    private static final String INTENT_TYPE = "vnd.android.cursor.item/event";
    private static final String INTENT_BEGIN_TIME = "beginTime";
    private static final String INTENT_ALL_DAY = "allDay";
    private static final String INTENT_END_TIME = "endTime";
    private static final String INTENT_TITLE = "title";
    private static final String INTENT_EVENT_LOCATION = "eventLocation";
    private static final String INTENT_DESCRIPTION = "description";

    private static final DateFormat dateTimeInstance = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);
    private static final DateFormat dateInstance = DateFormat.getDateInstance(DateFormat.MEDIUM);

    public CalendarResultHandler(Activity activity, CalendarParsedResult result) {
        super(activity, result);
    }

    @Override
    public CalendarParsedResult getResult() {
        return (CalendarParsedResult) super.getResult();
    }

    @Override
    public int getButtonCount() {
        return buttons.length;
    }

    @Override
    public int getButtonText(int index) {
        return buttons[index];
    }

    @Override
    public void handleButtonPress(int index) {
        if (index == ADD_TO_CALENDAR) {
            CalendarParsedResult calendarResult = getResult();

            String description = calendarResult.getDescription();
            String organizer = calendarResult.getOrganizer();
            if (organizer != null) { // No separate Intent key, put in description
                if (description == null) {
                    description = organizer;
                } else {
                    description = description + '\n' + organizer;
                }
            }

            addCalendarEvent(calendarResult.getSummary(),
                    calendarResult.getStart(),
                    calendarResult.isStartAllDay(),
                    calendarResult.getEnd(),
                    calendarResult.getLocation(),
                    description,
                    calendarResult.getAttendees());
        }
    }

    /**
     * Sends an intent to create a new calendar event by prepopulating the Add Event UI. Older
     * versions of the system have a bug where the event title will not be filled out.
     *
     * @param summary     A description of the event
     * @param start       The start time
     * @param allDay      if true, event is considered to be all day starting from start time
     * @param end         The end time (optional)
     * @param location    a text description of the event location
     * @param description a text description of the event itself
     * @param attendees   attendees to invite
     */
    private void addCalendarEvent(String summary,
                                  Date start,
                                  boolean allDay,
                                  Date end,
                                  String location,
                                  String description,
                                  String[] attendees) {
        Intent intent = new Intent(Intent.ACTION_INSERT);
        intent.setType(INTENT_TYPE);
        long startMilliseconds = start.getTime();
        intent.putExtra(INTENT_BEGIN_TIME, startMilliseconds);
        if (allDay) {
            intent.putExtra(INTENT_ALL_DAY, true);
        }
        long endMilliseconds;
        if (end == null) {
            if (allDay) {
                // + 1 day
                endMilliseconds = DateUtils.addDays(DateUtils.ceiling(start, Calendar.DATE), 1).getTime() - 1;
            } else {
                endMilliseconds = startMilliseconds;
            }
        } else {
            endMilliseconds = end.getTime();
        }
        intent.putExtra(INTENT_END_TIME, endMilliseconds);
        intent.putExtra(INTENT_TITLE, summary);
        intent.putExtra(INTENT_EVENT_LOCATION, location);
        intent.putExtra(INTENT_DESCRIPTION, description);
        if (attendees != null) {
            intent.putExtra(Intent.EXTRA_EMAIL, attendees);
            // Documentation says this is either a String[] or comma-separated String, which is right?
        }

        try {
            // Do this manually at first
            rawLaunchIntent(intent);
        } catch (ActivityNotFoundException anfe) {
            DebugLog.w("No calendar app available that responds to " + Intent.ACTION_INSERT);
            // For calendar apps that don't like "INSERT":
            intent.setAction(Intent.ACTION_EDIT);
            launchIntent(intent); // Fail here for real if nothing can handle it
        }
    }


    @Override
    public CharSequence getDisplayContents() {
        return null;
    }

    private static String format(boolean allDay, Date date) {
        if (date == null) {
            return null;
        }
        DateFormat format = allDay
                ? dateInstance
                : dateTimeInstance;
        return format.format(date);
    }

    @Override
    public String getDisplayTitle() {
        return getResult().getSummary();
    }

    @Override
    public String getDisplaySubtitle() {
        CalendarParsedResult calendarResult = getResult();
        StringBuilder builder = new StringBuilder(100);

        Date start = calendarResult.getStart();
        ParsedResult.maybeAppend(format(calendarResult.isStartAllDay(), start), builder);

        Date end = calendarResult.getEnd();
        if (end != null) {
            if (calendarResult.isEndAllDay() && DateUtils.isSameDay(start, DateUtils.addMilliseconds(end, -1))) {
                // Show only year/month/day
                // if it's all-day and this is the end date, it's exclusive, so show the user
                // that it ends on the day before to make more intuitive sense.
                // But don't do it if the event already (incorrectly?) specifies the same start/end
                return builder.toString();
            }
            builder.append(" - ");
            ParsedResult.maybeAppend(format(calendarResult.isEndAllDay(), end), builder);
        }
        return builder.toString();
    }

    @Override
    public String getDisplayText() {
        CalendarParsedResult calendarResult = getResult();
        StringBuilder builder = new StringBuilder();
        if (!TextUtils.isEmpty(calendarResult.getLocation())) {
            builder.append(calendarResult.getLocation());
        }
        if (!TextUtils.isEmpty(calendarResult.getDescription())) {
            builder.append("\n").append(calendarResult.getDescription());
        }
        return builder.toString();
    }

    @Override
    public int getDisplayIconRes() {
        return ThemesHelper.getResourceId(getActivity(), R.attr.ui_qr_ico_calendar);
    }
}
